package ru.yandex.travel.workflow;

import java.sql.Connection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.sql.DataSource;
import javax.validation.constraints.NotEmpty;

import com.google.common.collect.BiMap;
import com.google.protobuf.ProtocolMessageEnum;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.travel.hibernate.types.ProtobufEnumType;

@RequiredArgsConstructor
@Slf4j
public class ProtobufEnumDictionaryService implements InitializingBean {

    private static int PG_LOCK_ID = 1000;

    private final TransactionTemplate transactionTemplate;

    private final JdbcTemplate jdbcTemplate;

    private final Properties protobufEnumDictionaryProperties;

    @Override
    public void afterPropertiesSet() throws Exception {

        Connection connection = null;
        String databaseProductName = null;
        try {
            DataSource ds = jdbcTemplate.getDataSource();
            connection = ds.getConnection();
            databaseProductName = connection.getMetaData().getDatabaseProductName();
        } catch (Exception ex) {
            log.error("Error working with connection", ex);
        } finally {
            if (connection != null) {
                connection.close();
            }
        }

        final String dbProductName = databaseProductName;

        try {
            transactionTemplate.execute(ignored -> {
                boolean locked = false;

                if ("H2".equals(dbProductName)) {
                    log.info("Creating protobuf enum dictionary without locking");
                    locked = true;
                } else if ("PostgreSQL".equals(dbProductName)) {
                    boolean lockAcquired = jdbcTemplate.queryForObject(
                            "SELECT pg_try_advisory_lock(" + PG_LOCK_ID + ")",
                            (rs, rowNum) -> "t".equals(rs.getString("pg_try_advisory_lock")));
                    locked = lockAcquired;
                }
                if( !locked ) {
                    log.info("Skip creating protobuf enum dictionary as we couldn't obtain a lock");
                    return null;
                }


                String createTableStatement = String.format(
                        "CREATE TABLE IF NOT EXISTS %s " +
                                "(proto_class_name varchar, ordinal_value bigint, string_value varchar);",
                        protobufEnumDictionaryProperties.getTableName());

                jdbcTemplate.update(createTableStatement);

                String truncateTableStatement =
                        String.format("TRUNCATE TABLE %s;", protobufEnumDictionaryProperties.getTableName());

                jdbcTemplate.update(truncateTableStatement);

                Map<Class<? extends ProtocolMessageEnum>, BiMap<Integer, ? extends ProtocolMessageEnum>> mapping =
                        ProtobufEnumType.getGlobalMapping();

                List<Tuple3<String, Integer, String>> rowValues =
                        mapping.entrySet().stream().flatMap((e) ->
                                e.getValue().entrySet().stream().map(
                                        intToString -> Tuple3.tuple(e.getKey().getCanonicalName(), intToString.getKey(),
                                                intToString.getValue().toString()))
                        ).collect(Collectors.toList());

                String insertStatement = String.format(
                        "INSERT INTO %s(proto_class_name, ordinal_value,string_value) values (?,?,?);",
                        protobufEnumDictionaryProperties.getTableName()
                );

                rowValues.forEach(row -> {
                    jdbcTemplate.update(insertStatement, row._1, row._2, row._3);
                });

                return null;
            });
        } catch (Exception e) {
            log.error("Error creating protobuf enum dictionary", e);
        }
    }

    public String getTableName() {
        return protobufEnumDictionaryProperties.getTableName();
    }

    @Data
    public static class Properties {
        private boolean enabled;

        @NotEmpty
        private String tableName;
    }
}
