package ru.yandex.travel.workflow;

import java.time.Instant;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.Message;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.WorkflowEvent;
import ru.yandex.travel.workflow.repository.WorkflowEventRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@EnableConfigurationProperties(WorkflowEventQueueProperties.class)
public class WorkflowEventQueue {
    private final WorkflowRepository workflowRepository;
    private final WorkflowEventRepository workflowEventRepository;
    private final WorkflowEventQueueProperties workflowEventQueueProperties;
    private final AtomicLong pendingEventsCount;
    private final AtomicLong hangingEventsCount;
    private final ScheduledExecutorService scheduledExecutorService;
    private final TransactionTemplate transactionTemplate;

    public WorkflowEventQueue(WorkflowRepository workflowRepository, WorkflowEventRepository workflowEventRepository,
                              TransactionTemplate transactionTemplate,
                              WorkflowEventQueueProperties workflowEventQueueProperties) {
        this.workflowRepository = workflowRepository;
        this.workflowEventRepository = workflowEventRepository;
        this.workflowEventQueueProperties = workflowEventQueueProperties;
        this.pendingEventsCount = new AtomicLong(0);
        this.hangingEventsCount = new AtomicLong(0);
        this.scheduledExecutorService =
                Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).build());
        this.transactionTemplate = transactionTemplate;
        Gauge.builder("workflow.queue.pendingEventCount", pendingEventsCount::get).register(Metrics.globalRegistry);
        Gauge.builder("workflow.queue.hangingEventCount", hangingEventsCount::get).register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void start() {
        //TODO (mbobrov): make sure only master instance is publishing this stats
        scheduledExecutorService.scheduleAtFixedRate(this::refreshPendingEventsCount, 0,
                workflowEventQueueProperties.getStatsUpdateInterval().toMillis(), TimeUnit.MILLISECONDS);
    }

    @PreDestroy
    public void stop() {
        scheduledExecutorService.shutdown();
    }

    @TransactionMandatory
    public WorkflowEvent enqueueMessage(UUID workflowId, Message eventData) {
        return workflowEventRepository.saveAndFlush(WorkflowEvent.createEventFor(workflowId, eventData));
    }

    @TransactionMandatory
    public WorkflowEvent enqueueMessageNoFlush(UUID workflowId, Message eventData) {
        return workflowEventRepository.save(WorkflowEvent.createEventFor(workflowId, eventData));
    }

    @TransactionMandatory
    public Optional<WorkflowEvent> enqueueMessageOnlyIfRunning(UUID workflowId, Message eventData) {
        EWorkflowState workflowState = workflowRepository.getStateWithPessimisticForceIncrementLock(workflowId);
        if (workflowState == EWorkflowState.WS_RUNNING) {
            return Optional.of(enqueueMessage(workflowId, eventData));
        } else {
            return Optional.empty();
        }
    }

    public long getPendingEventsCount(UUID workflowId) {
        return workflowEventRepository.countByWorkflowIdAndStateEquals(workflowId, EWorkflowEventState.WES_NEW);
    }

    @TransactionMandatory
    public WorkflowEvent getEvent(UUID workflowId, Long eventId) {
        WorkflowEvent event = workflowEventRepository.getOne(eventId);
        Preconditions.checkState(event.getWorkflowId().equals(workflowId), "Inconsistent workflow ids in the argument" +
                " and in the database");
        return event;
    }

    @TransactionMandatory
    public Optional<WorkflowEvent> peekEvent(UUID workflowId) {
        return workflowEventRepository.findFirstByWorkflowIdAndStateOrderByCreatedAtAsc(workflowId,
                EWorkflowEventState.WES_NEW);
    }

    @TransactionMandatory
    public void dequeueAsProcessed(WorkflowEvent event) {
        event.setState(EWorkflowEventState.WES_PROCESSED);
        event.setProcessedAt(Instant.now());
    }

    @TransactionMandatory
    public void dequeueAsProcessingError(WorkflowEvent event) {
        event.setState(EWorkflowEventState.WES_CRASHED);
    }

    @TransactionMandatory
    public void registerForRetry(WorkflowEvent event) {
        event.setTimesTried(event.getTimesTried() + 1);
    }

    private void refreshPendingEventsCount() {
        try {
            transactionTemplate.execute(
                    ignored -> {
                        pendingEventsCount.set(workflowEventRepository.countPendingWorkflowEvents());
                        hangingEventsCount.set(workflowEventRepository.countHangingWorkflowEvents());
                        return null;
                    }
            );
        } catch (Exception e) {
            log.error("Error refreshing pending events count", e);
        }
    }

}
