package ru.yandex.travel.workflow;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.repository.WorkflowEventRepository;

@Slf4j
public class WorkflowEventStatePoller implements InitializingBean, DisposableBean {

    private final Set<Long> NO_EVENT_IDS = Set.of(-1L);

    private final TransactionTemplate transactionTemplate;

    private final WorkflowEventRepository workflowEventRepository;

    private final WorkflowEventStatusRegistry workflowEventStatusRegistry;

    private final ScheduledExecutorService poller;

    public WorkflowEventStatePoller(TransactionTemplate transactionTemplate,
                                    WorkflowEventRepository workflowEventRepository,
                                    WorkflowEventStatusRegistry workflowEventStatusRegistry) {
        this.transactionTemplate = transactionTemplate;
        this.workflowEventRepository = workflowEventRepository;
        this.workflowEventStatusRegistry = workflowEventStatusRegistry;
        this.poller = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder().setNameFormat("WorkflowEventStatePoller")
                        .setDaemon(true)
                        .build()
        );
    }

    @Override
    public void destroy() throws Exception {
        MoreExecutors.shutdownAndAwaitTermination(poller, 5, TimeUnit.SECONDS);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        poller.scheduleAtFixedRate(this::pollNotNewEventStatus, 0, 1000, TimeUnit.MILLISECONDS);
    }

    @TransactionMandatory
    public CompletableFuture<EWorkflowEventState> register(Long eventId) {
        return workflowEventStatusRegistry.register(eventId);
    }

    @TransactionMandatory
    public CompletableFuture<EWorkflowEventState> register(Long eventId, Executor executor) {
        return workflowEventStatusRegistry.register(eventId, executor);
    }


    private void pollNotNewEventStatus() {
        try {
            Set<Long> messageIds = workflowEventStatusRegistry.registeredMessages();
            if (messageIds.isEmpty()) {
                messageIds = NO_EVENT_IDS;
            }
            Set<Long> finalMessageIds = messageIds;
            List<Object[]> idStates =
                    transactionTemplate.execute(ignored -> {
                        return workflowEventRepository.pollEventStatus(finalMessageIds);
                    });
            if (idStates != null && !idStates.isEmpty()) {
                for (Object[] idState : idStates) {
                    Long id = (Long) idState[0];
                    EWorkflowEventState state = (EWorkflowEventState) idState[1];
                    if (state != EWorkflowEventState.WES_NEW) {
                        workflowEventStatusRegistry.resolve(id, state);
                    }
                }
            }
        } catch (Exception e) {
            log.error("Error polling event states", e);
        }
    }

}
