package ru.yandex.travel.workflow;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeoutException;

import lombok.Builder;
import lombok.Getter;

public class WorkflowEventStatusRegistry {

    @Builder(toBuilder = true)
    @Getter
    private final static class WorkflowExpirationHolder {
        private final Long id;
        private final Instant waitStatusTill;
    }

    private final Map<Long, CompletableFuture<EWorkflowEventState>> pendingFutures = new HashMap<>();

    private final Queue<WorkflowExpirationHolder> registered = new LinkedBlockingDeque<>();

    private final Clock clock;

    private final Duration maxWaitDuration;

    private final ExecutorService defaultExecutor;

    public WorkflowEventStatusRegistry(Clock clock,
                                       Duration maxWaitDuration,
                                       ExecutorService executorService) {
        this.maxWaitDuration = maxWaitDuration;
        this.defaultExecutor = executorService;
        this.clock = clock;
    }

    public synchronized CompletableFuture<EWorkflowEventState> register(Long messageId) {
        return register(messageId, defaultExecutor);
    }

    public synchronized CompletableFuture<EWorkflowEventState> register(Long messageId, Executor executor) {
        Instant expiration = Instant.now(clock).plus(maxWaitDuration.toMillis(), ChronoUnit.MILLIS);
        CompletableFuture<EWorkflowEventState> f = pendingFutures.computeIfAbsent(messageId, mid -> {
            CompletableFuture<EWorkflowEventState> result = new CompletableFuture<>();
            // hack to schedule future execution to a separate thread
            result.whenCompleteAsync((resultNotUsed, throwAbleIgnored) -> {
            }, executor);
            WorkflowExpirationHolder expirationHolder = WorkflowExpirationHolder.builder()
                    .id(messageId)
                    .waitStatusTill(expiration)
                    .build();
            registered.offer(expirationHolder);
            return result;
        });
        cleanExpired();
        return f;
    }

    public synchronized void resolve(Long messageId, EWorkflowEventState state) {
        CompletableFuture<EWorkflowEventState> f = pendingFutures.remove(messageId);
        if (f != null) {
            f.complete(state);
        }
        cleanExpired();
    }

    public synchronized Set<Long> registeredMessages() {
        return Set.copyOf(pendingFutures.keySet());
    }

    private void cleanExpired() {
        Instant now = Instant.now(clock);
        while (!registered.isEmpty() && registered.peek().waitStatusTill.isBefore(now)) {
            WorkflowExpirationHolder holder = registered.remove();
            CompletableFuture<EWorkflowEventState> f = pendingFutures.remove(holder.getId());
            if (f != null) {
                f.completeExceptionally(new TimeoutException(
                        String.format("Status of message with id %s hasn't changed, future expired", holder.getId())
                ));
            }
        }
    }
}
