package ru.yandex.travel.workflow;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.logging.EEventType;
import ru.yandex.travel.workflow.logging.TWorkflowLoggingEvent;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@RequiredArgsConstructor
public class WorkflowMaintenanceService {
    private final Map<EWorkflowState, EEventType> mapNewWfStateToLogEventType = ImmutableMap.of(
            EWorkflowState.WS_RUNNING, EEventType.ET_WORKFLOW_RESUMED,
            EWorkflowState.WS_PAUSED, EEventType.ET_WORKFLOW_PAUSED,
            EWorkflowState.WS_STOPPED, EEventType.ET_WORKFLOW_STOPPED,
            EWorkflowState.WS_CRASHED, EEventType.ET_WORKFLOW_CRASHED
    );
    private final WorkflowRepository workflowRepository;

    @TransactionMandatory
    public boolean pauseRunningWorkflow(UUID workflowId) {
        return changeWorkflowState(workflowId, EWorkflowState.WS_RUNNING, EWorkflowState.WS_PAUSED, false);
    }

    @TransactionMandatory
    public boolean resumePausedWorkflow(UUID workflowId) {
        return changeWorkflowState(workflowId, EWorkflowState.WS_PAUSED, EWorkflowState.WS_RUNNING, false);
    }

    @TransactionMandatory
    public boolean stopRunningWorkflow(UUID workflowId) {
        return changeWorkflowState(workflowId, EWorkflowState.WS_RUNNING, EWorkflowState.WS_STOPPED, false);
    }

    private boolean changeWorkflowState(UUID workflowId, EWorkflowState oldState, EWorkflowState newState,
                                        boolean ignoreLocking) {
        Workflow workflow = workflowRepository.getOne(workflowId);
        boolean changed = false;
        if (ignoreLocking) {
            int count = workflowRepository.compareAndSetWorkflowState(workflowId, oldState, newState);
            Preconditions.checkState(count <= 1, "Too many workflows affected");
            changed = count > 0;
        } else {
            if (workflow.getState() == oldState) {
                workflow.setState(newState);
                changed = true;
            }
        }
        if (changed) {
            AfterCommitEventLogging.logEvent(WorkflowProcessService.WF_EVENTS_LOGGER,
                    TWorkflowLoggingEvent.newBuilder()
                            .setType(mapNewWfStateToLogEventType.getOrDefault(newState, EEventType.ET_UNKNOWN))
                            .setWorkflowId(workflow.getId().toString())
                            .setEntityId(workflow.getEntityId().toString())
                            .setEntityType(workflow.getEntityType())
                            .setHappenedAt(ProtoUtils.fromInstant(Instant.now()))
                            .build());
        }
        return changed;
    }

    @TransactionMandatory
    public void pauseSupervisedRunningWorkflows(UUID workflowId) {
        changeSupervisedWorkflowsState(workflowId, EWorkflowState.WS_RUNNING, EWorkflowState.WS_PAUSED);
    }

    @TransactionMandatory
    public void resumeSupervisedPausedWorkflows(UUID workflowId) {
        changeSupervisedWorkflowsState(workflowId, EWorkflowState.WS_PAUSED, EWorkflowState.WS_RUNNING);
    }

    @TransactionMandatory
    public void stopSupervisedRunningWorkflows(UUID workflowId) {
        changeSupervisedWorkflowsState(workflowId, EWorkflowState.WS_RUNNING, EWorkflowState.WS_STOPPED);
    }

    @TransactionMandatory
    public void changeSupervisedWorkflowsState(UUID workflowId, EWorkflowState oldState, EWorkflowState newState) {
        List<UUID> supervisedWorkflows = workflowRepository.findSupervisedWorkflowsWithState(workflowId, oldState);
        if (supervisedWorkflows.isEmpty()) {
            return;
        }
        // TODO (mbobrov): think of updating them in cycle
        workflowRepository.batchCompareAndSetWorkflowState(supervisedWorkflows, oldState, newState);
        for (UUID pausedWorkflowId : supervisedWorkflows) {
            Workflow workflow = workflowRepository.getOne(pausedWorkflowId);
            AfterCommitEventLogging.logEvent(WorkflowProcessService.WF_EVENTS_LOGGER,
                    TWorkflowLoggingEvent.newBuilder()
                            .setType(mapNewWfStateToLogEventType.getOrDefault(newState, EEventType.ET_UNKNOWN))
                            .setWorkflowId(workflow.getId().toString())
                            .setEntityId(workflow.getEntityId().toString())
                            .setEntityType(workflow.getEntityType())
                            .setHappenedAt(ProtoUtils.fromInstant(Instant.now()))
                            .build()
            );
        }
    }
}
