package ru.yandex.travel.workflow;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.WorkflowEvent;

@Slf4j
@RequiredArgsConstructor
public class WorkflowMessageSender {

    private final WorkflowEventQueue workflowEventQueue;

    private final WorkflowEventStatePoller workflowEventStatePoller;

    @TransactionMandatory
    public void scheduleEvent(UUID workflowId, Message eventData) {
        Preconditions.checkNotNull(workflowId, "Workflow id must be not null");
        WorkflowEvent createdEvent = workflowEventQueue.enqueueMessage(workflowId, eventData);
        log.info("Enqueued event {} of class '{}' sent externally to workflow {}",
                createdEvent.getId(),
                createdEvent.getData().getClass().getSimpleName(),
                createdEvent.getWorkflowId());
    }

    @TransactionMandatory
    public CompletableFuture<EWorkflowEventState> scheduleEventWithTracking(UUID workflowId, Message eventData) {
        Preconditions.checkNotNull(workflowId, "Workflow id must be not null");
        WorkflowEvent createdEvent = workflowEventQueue.enqueueMessage(workflowId, eventData);
        CompletableFuture<EWorkflowEventState> eventFuture = workflowEventStatePoller.register(createdEvent.getId());
        log.info("Enqueued event {} of class '{}' sent externally to workflow {}",
                createdEvent.getId(),
                createdEvent.getData().getClass().getSimpleName(),
                createdEvent.getWorkflowId());
        return eventFuture;
    }

    @TransactionMandatory
    public void scheduleEventNoFlush(UUID workflowId, Message eventData) {
        Preconditions.checkNotNull(workflowId, "Workflow id must be not null");
        WorkflowEvent createdEvent = workflowEventQueue.enqueueMessageNoFlush(workflowId, eventData);
        log.info("Enqueued event {} of class '{}' sent externally to workflow {}",
                createdEvent.getId(),
                createdEvent.getData().getClass().getSimpleName(),
                createdEvent.getWorkflowId());
    }
}
