package ru.yandex.travel.workflow;

import javax.persistence.EntityManager;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.dao.support.PersistenceExceptionTranslator;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;

import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.tx.utils.SavepointHelper;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

/**
 * Event processing service
 * drains event queue for workflow
 * for each event a lock is obtained
 * processing of each event uses event handler matcher
 * event handler examples (state machine handler, reserve action handler )
 * after event handling new events are added to workflow queue
 * Release lock
 */
@Configuration
@EnableConfigurationProperties(WorkflowProcessingProperties.class)
@RequiredArgsConstructor
public class WorkflowProcessServiceConfiguration {

    private final WorkflowEventHandlerMatcher workflowEventHandlerMatcher;
    private final WorkflowRepository workflowRepository;
    private final WorkflowEventQueue workflowEventQueue;
    private final MessagingContextFactory messagingContextFactory;
    private final WorkflowProcessingProperties workflowProcessingProperties;
    @Autowired(required = false)
    private WorkflowProcessingListener workflowProcessingListener;

    @Bean
    @ConditionalOnMissingBean(WorkflowEventRetryStrategy.class)
    public WorkflowEventRetryStrategy workflowEventRetryStrategy() {
        return new DefaultWorkflowEventRetryStrategy();
    }

    @Bean
    public BatchPendingWorkflowsFetcher batchPendingWorkflowsFetcher(WorkflowRepository workflowRepository) {
        return new BatchPendingWorkflowsFetcher(workflowRepository);
    }

    @Bean
    public SequentialPendingWorkflowsFetcher sequentialPendingWorkflowsFetcher(WorkflowRepository workflowRepository) {
        return new SequentialPendingWorkflowsFetcher(workflowRepository);
    }

    @Bean
    public WorkflowProcessService workflowProcessService(ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper,
                                                         WorkflowEventRetryStrategy workflowEventRetryStrategy,
                                                         SavepointHelper savepointHelper,
                                                         BatchPendingWorkflowsFetcher batchPendingWorkflowsFetcher,
                                                         SequentialPendingWorkflowsFetcher sequentialPendingWorkflowsFetcher) {
        PendingWorkflowsFetcher fetcher = sequentialPendingWorkflowsFetcher;
        if (workflowProcessingProperties.getProcessingPools().getSchedulingMode() == WorkflowPoolSchedulingMode.BATCH) {
            fetcher = batchPendingWorkflowsFetcher;
        }
        if (workflowProcessingProperties.isUseSavepoints()) {
            return new WorkflowProcessServiceV3(
                    workflowEventHandlerMatcher, workflowRepository, workflowEventQueue, forcedRollbackTxManagerWrapper,
                    messagingContextFactory, workflowEventRetryStrategy, workflowProcessingListener,
                    savepointHelper, fetcher, workflowProcessingProperties);

        } else {
            return new WorkflowProcessServiceV2(
                    workflowEventHandlerMatcher, workflowRepository, workflowEventQueue, forcedRollbackTxManagerWrapper,
                    messagingContextFactory, workflowEventRetryStrategy, workflowProcessingListener,
                    fetcher, workflowProcessingProperties);
        }
    }

    @Bean
    public ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper(
            PlatformTransactionManager platformTransactionManager,
            PersistenceExceptionTranslator exceptionTranslator) {
        return new ForcedRollbackTxManagerWrapper(platformTransactionManager, exceptionTranslator);
    }

    @Bean
    public SavepointHelper savepointHelper(JpaTransactionManager transactionManager, EntityManager entityManager) {
        return new SavepointHelper(transactionManager, entityManager);
    }
}
