package ru.yandex.travel.workflow.base;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Map;
import java.util.function.BiConsumer;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.Message;

import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.MessagingUtils;
import ru.yandex.travel.workflow.WorkflowEventHandler;


public class AnnotatedWorkflowEventHandler<E> implements WorkflowEventHandler<E> {
    private final Map<Class<? extends Message>, BiConsumer<Message, MessagingContext<E>>> handlerMethods;

    @SuppressWarnings("unchecked")
    public AnnotatedWorkflowEventHandler() {
        AnnotatedWorkflowEventHandler<?> obj = this;
        ImmutableMap.Builder<Class<? extends Message>, BiConsumer<Message, MessagingContext<E>>> builder =
                ImmutableMap.builder();
        Arrays.stream(this.getClass().getMethods()).filter(m -> m.isAnnotationPresent(HandleEvent.class)).forEach(m -> {
            BiConsumer<Message, MessagingContext<E>> handler = (message, context) -> {
                try {

                    m.invoke(obj, message, context);
                } catch (IllegalAccessException e) {
                    throw new RuntimeException("Unable to access handler method", e);
                } catch (InvocationTargetException e) {
                    Throwable cause = e.getCause();
                    if (cause instanceof RuntimeException) {
                        throw (RuntimeException) cause;
                    } else if (cause instanceof Error) {
                        throw (Error) cause;
                    } else {
                        throw new RuntimeException(cause);
                    }
                }
            };
            validateHandlerParams(m);
            Class<? extends Message> messageType = m.getAnnotation(HandleEvent.class).value();
            if (messageType.equals(MessageTypeNotSet.class)) {
                // no explicit type, derive it from the parameters list
                messageType = (Class<? extends Message>) m.getParameterTypes()[0];
            }
            builder.put(messageType, handler);
        });
        handlerMethods = builder.build();
    }

    private static void validateHandlerParams(Method handlerMethod) {
        Class<?>[] parameterTypes = handlerMethod.getParameterTypes();
        Preconditions.checkArgument(parameterTypes.length == 2,
                "Exactly 2 handler method parameters are expected but got %s", parameterTypes.length);

        Class<?> messageType = parameterTypes[0];
        Preconditions.checkArgument(Message.class.isAssignableFrom(messageType),
                "The first handler parameter must be an instance of %s. Current type is %s",
                Message.class.getName(), messageType.getName());

        Class<?> messagingContextType = parameterTypes[1];
        Preconditions.checkArgument(MessagingContext.class.isAssignableFrom(messagingContextType),
                "The second handler parameter must be an instance of %s. Current type is %s",
                MessagingContext.class.getName(), messagingContextType.getName());
    }

    protected void handleDefault(Message event, MessagingContext<E> messagingContext) {
        MessagingUtils.throwOnUnmatchedEvent(event, messagingContext);
    }

    @Override
    public void handleEvent(Message event, MessagingContext<E> messagingContext) {
        BiConsumer<Message, MessagingContext<E>> handler = handlerMethods.get(event.getClass());
        if (handler != null) {
            handler.accept(event, messagingContext);
        } else {
            handleDefault(event, messagingContext);
        }
    }
}
