package ru.yandex.travel.workflow.base;

import java.time.Instant;
import java.util.Map;

import com.google.protobuf.Message;
import com.google.protobuf.ProtocolMessageEnum;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.workflow.AfterCommitEventLogging;
import ru.yandex.travel.workflow.BasicStateMessagingContext;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.StatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowEventHandler;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.entities.EntityStateTransition;
import ru.yandex.travel.workflow.entities.WorkflowEntity;
import ru.yandex.travel.workflow.logging.EEventType;
import ru.yandex.travel.workflow.logging.TWorkflowLoggingEvent;
import ru.yandex.travel.workflow.repository.EntityStateTransitionRepository;

@Slf4j
public class ProxyStateMapWorkflowEventHandler<S extends ProtocolMessageEnum, E extends WorkflowEntity<S>>
        implements WorkflowEventHandler<E> {

    private final Map<S, StatefulWorkflowEventHandler<S, E>> stateHandlers;

    private final EntityStateTransitionRepository entityStateTransitionRepository;

    public ProxyStateMapWorkflowEventHandler(Map<S, StatefulWorkflowEventHandler<S, E>> stateHandlers,
                                             EntityStateTransitionRepository entityStateTransitionRepository) {
        this.stateHandlers = stateHandlers;
        this.entityStateTransitionRepository = entityStateTransitionRepository;
    }

    @Override
    public void handleEvent(Message event, MessagingContext<E> messagingContext) {
        StateContext<S, E> stateContext = BasicStateMessagingContext.fromMessagingContext(messagingContext);
        S previousState = stateContext.getState();
        log.info("Handling event {} while in state {}",
                event.getClass().getSimpleName(),
                previousState);
        StatefulWorkflowEventHandler<S, E> handler = stateHandlers.get(stateContext.getState());
        if (handler == null) {
            throw new RuntimeException(
                    String.format("No handler for event %s in state %s; supported states: %s",
                            event.getClass().getSimpleName(), previousState, stateHandlers.keySet())
            );
        }
        handler.handleEvent(event, stateContext);
        if (stateContext.getState() != previousState) {
            log.info("State of workflow changed from {} to {}", previousState, stateContext.getState());

            EntityStateTransition stateTransition = new EntityStateTransition();
            WorkflowEntity<?> entity = messagingContext.getWorkflowEntity();
            stateTransition.setEntityId(entity.getId());
            stateTransition.setWorkflowId(messagingContext.getWorkflowId());
            stateTransition.setEntityType(entity.getEntityType());
            stateTransition.setFromState(previousState.getNumber());
            stateTransition.setToState(stateContext.getState().getNumber());
            stateTransition.setPreviousTransitionAt(entity.getLastTransitionAt());
            stateTransition.setTransitionAt(Instant.now());

            entityStateTransitionRepository.save(stateTransition);

            TWorkflowLoggingEvent loggingEvent = TWorkflowLoggingEvent.newBuilder()
                    .setType(EEventType.ET_ENTITY_STATE_TRANSITION)
                    .setEntityId(entity.getId().toString())
                    .setEntityType(entity.getEntityType())
                    .setWorkflowId(messagingContext.getWorkflowId().toString())
                    .setPreviousState(previousState.getValueDescriptor().getName())
                    .setPreviousStateTransitionAt(ProtoUtils.fromInstant(entity.getLastTransitionAt()))
                    .setNextState(stateContext.getState().getValueDescriptor().getName())
                    .setHappenedAt(ProtoUtils.fromInstant(stateTransition.getTransitionAt()))
                    .build();

            entity.setLastTransitionAt(stateTransition.getTransitionAt());
            AfterCommitEventLogging.logEvent(WorkflowProcessService.WF_ENTITY_EVENTS_LOGGER, loggingEvent);
        }

    }
}
