package ru.yandex.travel.workflow.entities;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.protobuf.Message;
import lombok.Data;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.workflow.AfterCommitEventLogging;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.logging.EEventType;
import ru.yandex.travel.workflow.logging.TWorkflowLoggingEvent;

@Entity
@Table(name = "workflows")
@Data
@BatchSize(size = 100)
public class Workflow {
    @Id
    private UUID id;

    private UUID supervisorId;

    private UUID entityId;

    private String entityType;

    @Type(type = "proto-enum")
    private EWorkflowState state;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    @OneToMany(mappedBy = "workflow", cascade = CascadeType.ALL)
    @BatchSize(size = 100)
    @OrderColumn(name = "transition_order_num")
    private List<WorkflowStateTransition> stateTransitions;

    private Instant sleepTill;

    private Integer workflowVersion;

    @Version
    private Integer version;

    private Integer processingPoolId;

    public void sleepFor(Duration duration) {
        sleepTill = Instant.now().plus(duration);
    }

    public static Workflow createWorkflowForEntity(WorkflowEntity<?> workflowEntity) {
        return createWorkflowForEntity(workflowEntity, (UUID) null);
    }

    public static Workflow createWorkflowForEntity(WorkflowEntity<?> workflowEntity,
                                                   WorkflowEntity<?> supervisorEntity) {
        Workflow supervisorWorkflow = supervisorEntity != null ? supervisorEntity.getWorkflow() : null;
        return createWorkflowForEntity(workflowEntity, supervisorWorkflow);
    }

    public static Workflow createWorkflowForEntity(WorkflowEntity<?> workflowEntity, Workflow supervisorWorkflow) {
        UUID supervisorId = supervisorWorkflow != null ? supervisorWorkflow.getId() : null;
        return createWorkflowForEntity(workflowEntity, supervisorId);
    }

    public static Workflow createWorkflowForEntity(WorkflowEntity<?> workflowEntity, UUID supervisorId) {
        return createWorkflowForEntity(workflowEntity, supervisorId, 0);
    }

    public static Workflow createWorkflowForEntity(WorkflowEntity<?> workflowEntity, UUID supervisorId, Integer workflowVariant) {
        Workflow result = new Workflow();
        result.setId(UUID.randomUUID());
        result.setSupervisorId(supervisorId);
        result.setEntityId(workflowEntity.getId());
        result.setEntityType(workflowEntity.getEntityType());
        result.setState(EWorkflowState.WS_RUNNING);
        result.setWorkflowVersion(workflowVariant);
        workflowEntity.setWorkflow(result);

        AfterCommitEventLogging.tryLogEvent(WorkflowProcessService.WF_ENTITY_EVENTS_LOGGER,
                TWorkflowLoggingEvent.newBuilder()
                        .setType(EEventType.ET_WORKFLOW_CREATED)
                        .setWorkflowId(result.getId().toString())
                        .setEntityId(result.getEntityId().toString())
                        .setEntityType(result.getEntityType())
                        .setHappenedAt(ProtoUtils.fromInstant(Instant.now()))
                        .build()
        );

        return result;
    }

    public void transitionTo(EWorkflowState newState, Message additionalData) {
        WorkflowStateTransition newTransition = new WorkflowStateTransition();
        newTransition.setFromState(state);
        newTransition.setToState(newState);
        newTransition.setData(additionalData);
        newTransition.setWorkflow(this);
        if (stateTransitions == null) {
            stateTransitions = new ArrayList<>();
        }
        stateTransitions.add(newTransition);
        state = newState;
    }
}
