package ru.yandex.travel.workflow.grpc;

import java.util.function.Function;

import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.StatusException;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.springframework.dao.ConcurrencyFailureException;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.grpc.CommonProtoChecks;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.workflow.WorkflowMaintenanceService;
import ru.yandex.travel.workflow.management.proto.TPauseWorkflowReq;
import ru.yandex.travel.workflow.management.proto.TPauseWorkflowRsp;
import ru.yandex.travel.workflow.management.proto.TResumeWorkflowReq;
import ru.yandex.travel.workflow.management.proto.TResumeWorkflowRsp;
import ru.yandex.travel.workflow.management.proto.WorkflowManagementServiceGrpc;

import static ru.yandex.travel.commons.grpc.ServerUtils.statusFromError;

@GrpcService
@Slf4j
@RequiredArgsConstructor
public class WorkflowManagementService extends WorkflowManagementServiceGrpc.WorkflowManagementServiceImplBase {

    private final WorkflowMaintenanceService workflowMaintenanceService;
    private final TransactionTemplate transactionTemplate;

    @Override
    public void pauseWorkflow(TPauseWorkflowReq request, StreamObserver<TPauseWorkflowRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, req -> {
            // TODO (mbobrov): add auth check by user id
            var workflowId = CommonProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            String userLogin = UserCredentials.get().getLogin();
            workflowMaintenanceService.pauseRunningWorkflow(workflowId);
            return TPauseWorkflowRsp.newBuilder().build();
        });
    }

    @Override
    public void resumeWorkflow(TResumeWorkflowReq request, StreamObserver<TResumeWorkflowRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, req -> {
            var workflowId = CommonProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            String userLogin = UserCredentials.get().getLogin();
            workflowMaintenanceService.resumePausedWorkflow(workflowId);
            return TResumeWorkflowRsp.newBuilder().build();
        });
    }

    private <ReqT, RspT> void synchronouslyWithTx(ReqT request, StreamObserver<RspT> observer,
                                                  Function<ReqT, RspT> handler) {
        ServerUtils.synchronously(log, request, observer,
                rq -> transactionTemplate.execute(ignored -> handler.apply(rq)),
                ex -> mapStatusException(log, request, ex)
        );
    }

    public static <ReqT> StatusException mapStatusException(Logger log, ReqT request, Throwable ex) {
        log.error("Caught exception {} while handling request {}",
                ex.getClass().getSimpleName(), request.getClass().getSimpleName(), ex);
        TError error = ProtoUtils.errorFromThrowable(ex, false);
        if (ex instanceof ConcurrencyFailureException) {
            error = error.toBuilder().setCode(EErrorCode.EC_ABORTED).build();
        }
        // TODO(tivelkov): passing verbose=true causes stack trace to get into headers, thus causing
        //  'Header size exceeded max allowed size'
        Status status = statusFromError(error);
        Metadata trailers = new Metadata();
        trailers.put(ServerUtils.METADATA_ERROR_KEY, error);
        return status.asException(trailers);
    }
}
