package ru.yandex.travel.workflow.ha;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicBoolean;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.workflow.WorkflowProcessService;

@Slf4j
public class MasterAwareResourceManager implements MasterStatusAwareResource {

    private final WorkflowProcessService workflowProcessService;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final ImmutableSet<TaskProcessor<?>> taskProcessors;

    private final AtomicBoolean isMaster;

    public MasterAwareResourceManager(WorkflowProcessService workflowProcessService,
                                      ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper,
                                      Collection<TaskProcessor<?>> taskProcessors) {
        this.workflowProcessService = workflowProcessService;
        this.forcedRollbackTxManagerWrapper = forcedRollbackTxManagerWrapper;
        this.taskProcessors = ImmutableSet.copyOf(taskProcessors);
        this.isMaster = new AtomicBoolean(false);
    }

    @Override
    public void promotedToMaster() {
        log.info("Resuming work as a promotion to master happened");
        forcedRollbackTxManagerWrapper.resumeCommits();
        workflowProcessService.resume();
        taskProcessors.forEach(taskProcessor -> {
            // we're extra cautious doing our best to resume all task processors
            try {
                log.info("Resuming task processor: {}", taskProcessor.getName());
                taskProcessor.resume();
            } catch (Exception e) {
                log.error("Error resuming task processor {}", taskProcessor.getName(), e);
            }
        });
        isMaster.set(true);
    }

    @Override
    public void prepareToStandby() {
        isMaster.set(false);
        workflowProcessService.pauseAll();
        taskProcessors.forEach(taskProcessor -> {
            try {
                log.info("Pausing task processor: {}", taskProcessor.getName());
                taskProcessor.pause();
            } catch (Exception e) {
                log.error("Error resuming task processor {}", taskProcessor.getName(), e);
            }
        });
        forcedRollbackTxManagerWrapper.pauseCommits();
    }

    @Override
    public void forceStandby() {
        Preconditions.checkState(!isMaster.get(), "Forced stop must not happen in case the node is master. That's a " +
                "last resort method if prepareToStandby didn't work properly");
        forcedRollbackTxManagerWrapper.pauseCommits();
    }

    @Override
    public void stopAll() {
        Preconditions.checkState(!isMaster.get(), "Stop all must not happen in case the node is master. " +
                "That's a clean up method");
        try {
            workflowProcessService.stop();
        } catch (Exception e) {
            log.error("Error stopping workflow", e);
        }
        taskProcessors.forEach(taskProcessor -> {
            try {
                log.info("Stopping task processor: {}", taskProcessor.getName());
                taskProcessor.shutdown();
            } catch (Exception e) {
                log.error("Error resuming task processor {}", taskProcessor.getName(), e);
            }
        });
    }

    public boolean isMaster() {
        return isMaster.get();
    }
}
