package ru.yandex.travel.workflow.single_operation;

import java.util.UUID;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorHelper;
import ru.yandex.travel.task_processor.TaskProcessorProperties;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.repository.SingleOperationRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Configuration
@ConditionalOnProperty("single-operation.enabled")
public class SingleOperationConfiguration {

    @Bean
    @ConditionalOnMissingBean(SingleOperationRunnerProvider.class)
    public SingleOperationRunnerProvider emptySingleOperationRunnerProvider() {
        return operationType -> {
            throw new IllegalStateException("You must configure single operation runner provider " +
                    "in order to use single operations in your application");
        };
    }

    @Bean
    @ConditionalOnMissingBean(SingleOperationSupervisorProvider.class)
    public SingleOperationSupervisorProvider nullSupervisorProvider() {
        return operationType -> null;
    }

    @Bean
    public SingleOperationService singleOperationService(
            WorkflowMessageSender workflowMessageSender,
            WorkflowRepository workflowRepository,
            SingleOperationRepository singleOperationRepository,
            SingleOperationSupervisorProvider singleOperationSupervisorProvider
    ) {
        return new SingleOperationService(singleOperationRepository, singleOperationSupervisorProvider,
                workflowRepository, workflowMessageSender);
    }

    @Bean
    public SingleOperationWorkflowHandler singleOperationWorkflowHandler(
            SingleOperationRunnerProvider singleOperationRunnerProvider
    ) {
        return new SingleOperationWorkflowHandler(singleOperationRunnerProvider);
    }

    @ConfigurationProperties(value = "single-operation.schedule", ignoreUnknownFields = false)
    @Bean
    public TaskProcessorProperties singleOperationSchedulingTaskProcessorProperties() {
        return TaskProcessorProperties.builder()
                .name("SingleOperationScheduler")
                .daemonPoolThreads(true)
                .gracefulPoolShutdown(true)
                .build();
    }

    @Bean
    public TaskProcessor<UUID> singleOperationSchedulingTaskProcessor(
            @Qualifier("singleOperationSchedulingTaskProcessorProperties") TaskProcessorProperties properties,
            SingleOperationService singleOperationService,
            ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper
    ) {
        TaskKeyProvider<UUID> taskKeyProvider = TaskProcessorHelper.createTaskKeyProvider(
                singleOperationService::getOperationsToSchedule,
                singleOperationService::getPendingOperationsCount
        );
        return new TaskProcessor<>(taskKeyProvider,
                singleOperationService::scheduleOperationCommit,
                forcedRollbackTxManagerWrapper,
                TaskProcessorHelper.createDefaultTxDefinition("singleOperationSchedulingTaskTxDef"),
                properties
        );
    }
}
