package ru.yandex.travel.workflow.single_operation;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.SingleOperation;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.SingleOperationRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;
import ru.yandex.travel.workflow.single_operation.proto.ESingleOperationState;
import ru.yandex.travel.workflow.single_operation.proto.TCommit;

@RequiredArgsConstructor
@Slf4j
public class SingleOperationService {

    private final SingleOperationRepository singleOperationRepository;

    private final SingleOperationSupervisorProvider singleOperationSupervisorProvider;

    private final WorkflowRepository workflowRepository;

    private final WorkflowMessageSender workflowMessageSender;

    @TransactionMandatory
    public UUID runOperation(String name, String operationType, Object input) {
        return scheduleSingleOperation(name, null, operationType, input, null);
    }

    @TransactionMandatory
    public UUID runUniqueOperation(String uniqueName, String operationType, Object input) {
        return safeScheduleUniqueOperation(uniqueName, operationType, input, null);
    }

    @TransactionMandatory
    public UUID runUniqueOperationUnsafe(String uniqueName, String operationType, Object input) {
        return scheduleSingleOperation(uniqueName, uniqueName, operationType, input, null);
    }

    @TransactionMandatory
    public UUID scheduleOperation(String name, String operationType, Object input, Instant scheduleAt) {
        return scheduleSingleOperation(name, null, operationType, input, scheduleAt);
    }

    @TransactionMandatory
    public UUID scheduleUniqueOperation(String uniqueName, String operationType, Object input, Instant scheduleAt) {
        return safeScheduleUniqueOperation(uniqueName, operationType, input, scheduleAt);
    }

    @TransactionMandatory
    public UUID scheduleUniqueOperationUnsafe(String uniqueName, String operationType, Object input, Instant scheduleAt) {
        return scheduleSingleOperation(uniqueName, uniqueName, operationType, input, scheduleAt);
    }

    @TransactionMandatory
    public void scheduleOperationCommit(UUID operationId) {
        SingleOperation operation = singleOperationRepository.getOne(operationId);
        if (operation.isCommitSent()) {
            log.info("Commit already sent to operation {}", operationId);
            return;
        }
        workflowMessageSender.scheduleEvent(operation.getWorkflow().getId(), TCommit.getDefaultInstance());
        operation.setCommitSent(true);
    }

    @TransactionMandatory
    public List<UUID> getOperationsToSchedule(Set<UUID> excludeIds, int fetchSize) {
        Pageable pageable = PageRequest.of(0, fetchSize);
        Set<UUID> exclude = excludeIds != null && !excludeIds.isEmpty() ? excludeIds :
                SingleOperationRepository.NO_EXCLUDE_IDS;
        return singleOperationRepository.findIdsToSchedule(Instant.now(), exclude, pageable);
    }

    @TransactionMandatory
    public Long getPendingOperationsCount(Set<UUID> excludeIds) {
        Set<UUID> exclude = excludeIds != null && !excludeIds.isEmpty() ? excludeIds :
                SingleOperationRepository.NO_EXCLUDE_IDS;
        return singleOperationRepository.countOperationsToSchedule(Instant.now(), exclude);
    }

    @TransactionMandatory
    public boolean hasOperationsWithName(String name) {
        return singleOperationRepository.findAllByUniqueName(name).size() > 0;
    }

    @TransactionMandatory
    public <T> T getOperationResultByUniqueName(String name, Class<T> valueType) {
        var operation = getSingleOperationByUniqueName(name);
        Preconditions.checkState(operation.getState() == ESingleOperationState.ERS_SUCCESS);
        try {
            return SingleOperationJsonMapper.INSTANCE.treeToValue(operation.getOutput(), valueType);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error parsing json payload provided by operation", e);
        }
    }

    @TransactionMandatory
    public SingleOperation getSingleOperationByUniqueName(String name) {
        var allOperations = singleOperationRepository.findAllByUniqueName(name);
        Preconditions.checkState(allOperations.size() == 1, "Expected exactly one operation");
        return allOperations.get(0);
    }

    private UUID safeScheduleUniqueOperation(String uniqueName,
                                             String operationType,
                                             Object payload,
                                             Instant instant) {
        if (!Strings.isNullOrEmpty(uniqueName) && hasOperationsWithName(uniqueName)) {
            return null;
        } else {
            return scheduleSingleOperation(uniqueName, uniqueName, operationType, payload, instant);
        }
    }

    private UUID scheduleSingleOperation(String name, String uniqueName, String operationType, Object payload, Instant instant) {
        SingleOperation singleOperation = new SingleOperation();
        UUID uuid = UUID.randomUUID();
        singleOperation.setId(uuid);
        singleOperation.setUniqueName(Strings.isNullOrEmpty(uniqueName) ? uuid.toString() : uniqueName);
        singleOperation.setName(Strings.isNullOrEmpty(name) ? uuid.toString() : name);
        singleOperation.setOperationType(operationType);
        singleOperation.setInput(SingleOperationJsonMapper.INSTANCE.valueToTree(payload));
        singleOperation.setState(ESingleOperationState.ERS_NEW);

        singleOperation = singleOperationRepository.save(singleOperation);
        Workflow workflow = workflowRepository.save(Workflow.createWorkflowForEntity(
                singleOperation, singleOperationSupervisorProvider.supervisorFor(operationType)
        ));

        if (instant != null) {
            singleOperation.setScheduledAt(instant);
            singleOperation.setCommitSent(false);
        } else {
            workflowMessageSender.scheduleEventNoFlush(workflow.getId(), TCommit.getDefaultInstance());
            singleOperation.setCommitSent(true);
        }

        return singleOperation.getId();
    }

}
