package ru.yandex.travel.workflow.single_operation;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.entities.SingleOperation;
import ru.yandex.travel.workflow.single_operation.proto.ESingleOperationState;
import ru.yandex.travel.workflow.single_operation.proto.TCommit;

@RequiredArgsConstructor
@Slf4j
public class SingleOperationWorkflowHandler extends AnnotatedWorkflowEventHandler<SingleOperation> {

    private final SingleOperationRunnerProvider singleOperationRunnerProvider;

    @HandleEvent
    @SuppressWarnings("unchecked")
    public void handleCommit(TCommit event, MessagingContext<SingleOperation> ctx) {
        SingleOperation operation = ctx.getWorkflowEntity();
        Preconditions.checkState(operation.getState() == ESingleOperationState.ERS_NEW, "Operation must be in ERS_NEW" +
                " state");
        var runner = singleOperationRunnerProvider.runnerForOperationType(operation.getOperationType());
        Preconditions.checkState(runner != null, "Runner of type %s for operation %s not found",
                operation.getOperationType(), operation.getName());
        log.info("Starting single operation {} of type {}", operation.getName(), operation.getOperationType());
        try {
            Object input = SingleOperationJsonMapper.INSTANCE.treeToValue(
                    operation.getInput(), runner.getInputClass()
            );
            Object output = runner.runOperation(input);
            operation.setOutput(SingleOperationJsonMapper.INSTANCE.valueToTree(output));
            operation.setState(ESingleOperationState.ERS_SUCCESS);
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Error parsing json payload provided by operation", e);
        }
        log.info("Finished single operation {} of type {}", operation.getName(), operation.getOperationType());
    }
}
