package ru.yandex.travel.yt_lucene_index;

import java.util.Collections;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.protobuf.GeneratedMessageV3;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Parser;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.StoredField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BasePersistentConfig<ProtoClass extends GeneratedMessageV3> {
    private static final String FIELD_PROTO_BYTES = "p";

    private final YtLuceneIndex luceneIndex;

    protected final String name;
    protected final Logger log;

    public BasePersistentConfig(YtLuceneIndexParams params, String name, Supplier<ProtoClass.Builder> builderSupplier) {
        this.name = name;
        this.log = LoggerFactory.getLogger(this.getClass().getName() + "[" + name + "]");
        int indexStructureVersion = ProtobufIndexVersionBuilder.build(builderSupplier.get());
        luceneIndex = new YsonYtLuceneIndex(params, name, indexStructureVersion, (row) -> {
            ProtoClass.Builder builder = builderSupplier.get();
            ProtoFiller.fillByYtNode(row, builder, "");
            byte[] protoData = builder.build().toByteArray();
            Document document = new Document();
            document.add(new StoredField(FIELD_PROTO_BYTES, protoData));
            return Collections.singletonList(document);
        });
        luceneIndex.setKeepIndexOpened(false);
    }

    protected void setIndexUpdateHandler(YtLuceneIndex.IndexUpdateHandler indexUpdateHandler) {
        luceneIndex.setIndexUpdateHandler(indexUpdateHandler);
    }


    protected void forEachProtoRecord(Parser<ProtoClass> parser, Consumer<ProtoClass> consumer) {
        luceneIndex.forEachDocument(doc -> {
            try {
                consumer.accept(parser.parseFrom(doc.getField(FIELD_PROTO_BYTES).binaryValue().bytes));
            } catch (InvalidProtocolBufferException e) {
                throw new RuntimeException(e);
            }
        });
    }

    @PostConstruct
    public void start() {
        luceneIndex.start();
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void stop() {
        luceneIndex.stop();
    }

    public boolean isReady() {
        return luceneIndex.isReady();
    }
}
