package ru.yandex.travel.yt_lucene_index;

import java.io.IOException;
import java.nio.file.Path;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;

import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.custom.CustomAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.SearcherFactory;
import org.apache.lucene.search.SearcherManager;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.MMapDirectory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtConfiguration;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.common.YtException;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.misc.io.RuntimeIoException;


public class BaseYtLuceneIndex<TYtRowType> implements YtLuceneIndex {
    private static final String FIELD_SYSTEM_DOCUMENT = "systemDocument";
    private static final String FIELD_MOD_TIME = "modTime";
    private static final String FIELD_STRUCTURE_VERSION = "structureVersion";
    private static final long MAX_UPDATE_ERRORS = 5;

    private final Directory indexDirectory;
    private final YtLuceneIndexParams params;
    private final String name;
    private final int structureVersion;
    private final Function<TYtRowType, Iterable<Document>> documentProducer;
    private final YTableEntryType<TYtRowType> ytEntryType;
    private final Logger log;
    private final Map<String, Yt> yts;
    private final ScheduledExecutorService executorService;
    private final Path indexPath;
    private final AtomicBoolean isReady = new AtomicBoolean(false);
    private final AtomicBoolean isUpdateError = new AtomicBoolean(false);
    private final AtomicInteger updateErrorCounter = new AtomicInteger(0);
    private final AtomicReference<SearcherManager> searcherManagerRef = new AtomicReference<>();
    private final AtomicReference<IndexUpdateHandler> indexUpdateHandler = new AtomicReference<>();
    private boolean keepIndexOpened;
    private String localModTime;
    private AtomicLong lastUpdateTimeMs = new AtomicLong(0);
    private AtomicInteger indexNumDocs = new AtomicInteger(0);

    private final Timer searchTimeMeter;
    private final Counter searchEventMeter;

    public BaseYtLuceneIndex(YtLuceneIndexParams params, String name, int structureVersion,
                             Function<TYtRowType, Iterable<Document>> documentProducer,
                             YTableEntryType<TYtRowType> ytEntryType) {
        this.params = params;
        this.name = name;
        this.structureVersion = structureVersion;
        this.documentProducer = documentProducer;
        this.ytEntryType = ytEntryType;

        this.log = LoggerFactory.getLogger(this.getClass().getName() + "[" +name + "]");
        this.yts = new HashMap<>();
        this.executorService = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder()
                        .setNameFormat("yt-lucene-index-thread-" + name + "-%d")
                        .build()
        );
        this.keepIndexOpened = true;
        if (params.getProxy().isEmpty()) {
            throw new IllegalArgumentException("Proxy list should not be empty");
        }
        indexPath = Path.of(params.getIndexPath()).toAbsolutePath().normalize();

        try {
            MMapDirectory mMapDirectory = (MMapDirectory)MMapDirectory.open(indexPath);
            mMapDirectory.setPreload(params.isPreload());
            indexDirectory = mMapDirectory;
        } catch (IOException e) {
            log.error("Failed to open lucene index directory {}", indexPath, e);
            throw new RuntimeException(e);
        }

        for (String proxy: params.getProxy()) {
            log.info("Will use proxy {}", proxy);
            YtConfiguration configuration = YtConfiguration.builder()
                    .withSimpleCommandsRetries(3)
                    .withHeavyCommandsRetries(3)
                    .withToken(params.getToken())
                    .withApiHost(proxy)
                    .build();
            this.yts.put(proxy, YtUtils.http(configuration));
        }
        Gauge.builder("yt.lucene.numDocs", this::getIndexDocumentCount)
                .tag("name", name)
                .register(Metrics.globalRegistry);
        Gauge.builder("yt.lucene.timeSinceLastUpdateSec", this::getSecondsSinceLastUpdate)
                .tag("name", name)
                .register(Metrics.globalRegistry);
        Gauge.builder("yt.lucene.updateError", this::getUpdateError)
                .tag("name", name)
                .register(Metrics.globalRegistry);
        searchTimeMeter = Timer.builder("yt.lucene.searchTime")
                .tag("name", name)
                .serviceLevelObjectives(List.of(1, 5, 10, 25, 50, 100).stream().map(Duration::ofMillis).toArray(Duration[]::new))
                .register(Metrics.globalRegistry);
        searchEventMeter = Counter.builder("yt.lucene.search")
                .tag("name", name)
                .register(Metrics.globalRegistry);
    }

    @Override
    public void setIndexUpdateHandler(IndexUpdateHandler indexUpdateHandler1) {
        indexUpdateHandler.set(indexUpdateHandler1);
    }

    @Override
    public void setKeepIndexOpened(boolean keepIndexOpened) {
        this.keepIndexOpened = keepIndexOpened;
    }

    @Override
    public void start() {
        try {
            reopenIndex();
        } catch (Exception e) {
            log.warn("Failed to open index at startup: {}", e.getMessage());
        }
        executorService.scheduleAtFixedRate(
                this::checkForUpdates, 0, params.getUpdateInterval().toMillis(), TimeUnit.MILLISECONDS);
    }

    @Override
    public void stop() {
        log.debug("Stop started");
        MoreExecutors.shutdownAndAwaitTermination(executorService, params.getShutdownTimeout().toNanos(),
                TimeUnit.NANOSECONDS);
        closeSearcherManager();
        try {
            log.info("Closing IndexDirectory");
            indexDirectory.close();
        } catch (IOException e) {
            log.error("Failed to close IndexDirectory", e);
        }
        log.info("Stop finished");
    }

    private void closeSearcherManager() {
        SearcherManager sm = searcherManagerRef.getAndSet(null);
        if (sm != null) {
            log.info("Closing SearcherManager");
            try {
                sm.close();
            } catch (IOException e) {
                log.error("Failed to close SearcherManager", e);
            }
        }
    }

    private <T> T search(SearchExecutor<T> executor, boolean withTimeMeasure) {
        long started = System.currentTimeMillis();
        searchEventMeter.increment();
        SearcherManager sm = searcherManagerRef.get();
        if (sm == null) {
            throw new IllegalStateException("Lucene index '" + name + "' is not ready");
        }
        IndexSearcher searcher = null;
        try {
            searcher = sm.acquire();
            return executor.execute(searcher);
        } catch (IOException e) {
            throw new IllegalStateException("Index '" + name + "' is not readable", e);
        } finally {
            try {
                sm.release(searcher);
            } catch (IOException e) {
                log.error("Failed to release IndexSearcher", e);
            }
            if (withTimeMeasure) {
                long duration = System.currentTimeMillis() - started;
                searchTimeMeter.record(duration, TimeUnit.MILLISECONDS);
            }
        }
    }

    @Override
    public <T> T search(SearchExecutor<T> executor) {
        return search(executor, true);
    }

    @Override
    public void forEachDocument(Consumer<Document> consumer) {
        search((searcher) -> {
            TopDocs topDocs = searcher.search(new MatchAllDocsQuery(), searcher.getIndexReader().numDocs());
            for (ScoreDoc scoreDoc : topDocs.scoreDocs) {
                Document doc = searcher.doc(scoreDoc.doc);
                if (doc.getField(FIELD_SYSTEM_DOCUMENT) == null) {
                    consumer.accept(doc);
                }
            }
            return null;
        }, false);
    }

    @Override
    public boolean isReady() {
        return isReady.get();
    }

    private void reopenIndex() {
        long started = System.currentTimeMillis();
        log.info("Reopening index at {}", indexPath);
        IndexSearcher searcher = null;
        SearcherManager sm = null;
        try {
            sm = searcherManagerRef.get();
            if (sm == null) {
                sm = new SearcherManager(indexDirectory, new SearcherFactory());
            } else {
                sm.maybeRefreshBlocking();
            }
            searcher = sm.acquire();
            Query query = LongPoint.newExactQuery(FIELD_SYSTEM_DOCUMENT, 1);
            TopDocs topDocs = searcher.search(query, 1);
            if (topDocs.totalHits < 1) {
                throw new IllegalStateException("Failed to find system document in index");
            }
            Document systemDocument = searcher.doc(topDocs.scoreDocs[0].doc);
            int docStructureVersion = systemDocument.getField(FIELD_STRUCTURE_VERSION).numericValue().intValue();
            if (docStructureVersion != structureVersion) {
                throw new IllegalStateException(String.format("Wrong index structure version: got %s, expected %s",
                        docStructureVersion, structureVersion));
            }
            String modTime = systemDocument.getField(FIELD_MOD_TIME).stringValue();
            if (modTime == null) {
                throw new IllegalStateException(String.format("System document has no %s field", FIELD_MOD_TIME));
            }
            indexNumDocs.set(searcher.getIndexReader().numDocs() - 1); // 1 is system doc
            setLocalModTime(modTime);
            searcherManagerRef.set(sm);
            log.info("Index reopened in {} ms, has {} documents. Modification time is {}",
                    System.currentTimeMillis() - started, indexNumDocs.get(), modTime);
            try {
                IndexUpdateHandler updateHandler = indexUpdateHandler.get();
                if (updateHandler != null) {
                    updateHandler.process();
                }
            } catch (Exception e) {
                log.error("Exception during index update handler", e);
                throw e;
            }
            isReady.set(true);
            if (!keepIndexOpened) {
                log.info("Closing index");
                closeSearcherManager();
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (sm != null && searcher != null) {
                try {
                    sm.release(searcher);
                } catch (IOException e) {
                    log.error("Failed to release IndexSearcher", e);
                }
            }
        }
    }

    private void handleYtUpdateError() {
        if (!isReady.get()) {
            // Плохо если данных нет, и прочитать не можем
            isUpdateError.set(true);
        }
    }

    private void checkForUpdates() {
        YPath path;
        try {
            path = YPath.simple(params.getTablePath());
        } catch (Exception e) {
            log.error("Failed to construct YPath from '{}'", params.getTablePath(), e);
            isUpdateError.set(true);
            return;
        }
        Yt selectedYt = null;
        String selectedYtProxy = null;
        String maxModTime = null;
        for (Map.Entry<String, Yt> entry: yts.entrySet()) {
            try {
                String remoteModTime = entry.getValue().cypress().get(path.attribute("modification_time")).stringValue();
                if (maxModTime == null || (remoteModTime.compareTo(maxModTime) > 0)) {
                    maxModTime = remoteModTime;
                    selectedYtProxy = entry.getKey();
                    selectedYt = entry.getValue();
                }
            } catch (Exception e) {
                log.error("Failed to check mod time at YT Proxy '{}'", entry.getKey(), e);
            }
        }
        if (selectedYt == null) {
            log.warn("Failed to check remote mod time, no availiable YTs");
            handleYtUpdateError();
            return;
        }
        if (maxModTime.equals(localModTime)) {
            log.info("Skip loading from YT table, because modtime not changed {}", localModTime);
            return;
        }
        log.info("Loading new data from YT {}.`{}`, modTime is {}", selectedYtProxy, params.getTablePath(), maxModTime);

        try (Analyzer analyzer = CustomAnalyzer.builder().withTokenizer("standard").build();
             IndexWriter indexWriter = new IndexWriter(indexDirectory, new IndexWriterConfig(analyzer)
                     .setOpenMode(IndexWriterConfig.OpenMode.CREATE)
                     .setCommitOnClose(false))) {
            Document systemDoc = new Document();
            systemDoc.add(new LongPoint(FIELD_SYSTEM_DOCUMENT, 1)); // For searching
            systemDoc.add(new StoredField(FIELD_SYSTEM_DOCUMENT, 1));// For checking in getAllDocuments
            systemDoc.add(new StoredField(FIELD_MOD_TIME, maxModTime));
            systemDoc.add(new StoredField(FIELD_STRUCTURE_VERSION, structureVersion));
            indexWriter.addDocument(systemDoc);
            AtomicInteger loadedRows = new AtomicInteger(0);
            AtomicInteger loadedDocs = new AtomicInteger(0);
            selectedYt.tables().read(path, ytEntryType, (row) -> {
                try {
                    for (Document doc : documentProducer.apply(row)) {
                        loadedDocs.incrementAndGet();
                        indexWriter.addDocument(doc);
                    }
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
                if (loadedRows.incrementAndGet() % 10000 == 0) {
                    log.info("Loading from YT: {} rows and {} documents already loaded", loadedRows.get(), loadedDocs.get());
                }
            });
            log.info("Loaded from YT: {} rows and {} documents", loadedRows.get(), loadedDocs.get());
            log.debug("Merging index");
            indexWriter.forceMerge(1);
            log.debug("Commiting");
            indexWriter.commit();
        } catch (YtException e) {
            log.error("Loading from YT Proxy '{}' failed due to YT Error", selectedYtProxy, e);
            handleYtUpdateError();
            return;
        } catch (RuntimeIoException e) {
            log.error("Loading from YT Proxy '{}' failed due to I/O Error", selectedYtProxy, e);
            handleYtUpdateError();
            return;
        } catch (Exception e) {
            // Two options:
            // 1. Lucene exception, including all exceptions from documentProducer -> it should cause alerts
            // 2. Sporadic YT Runtime exceptions -> it should NOT cause alerts
            // We cannot reliably distinguish these situations, so rely on counter
            log.error("Loading from YT Proxy '{}' failed", selectedYtProxy, e);
            if (!isReady.get() || updateErrorCounter.incrementAndGet() > MAX_UPDATE_ERRORS) {
                isUpdateError.set(true);
            }
            return;
        }
        try {
            reopenIndex();
        } catch (Exception e) {
            log.error("Failed to (re)open index after index update", e);
            isUpdateError.set(true);
            return;
        }
        isUpdateError.set(false);
        updateErrorCounter.set(0);
    }

    private int getIndexDocumentCount() {
        return indexNumDocs.get();
    }

    private Long getSecondsSinceLastUpdate() {
        long v = lastUpdateTimeMs.get();
        if (v != 0) {
            return (System.currentTimeMillis() - v) / 1000;
        } else {
            return null;
        }
    }

    private void setLocalModTime(String modTime) {
        localModTime = modTime;
        lastUpdateTimeMs.set(ZonedDateTime.parse(modTime).toInstant().toEpochMilli());
    }

    private Integer getUpdateError() {
        return isUpdateError.get() ? 1 : 0;
    }
}
