package ru.yandex.travel.yt_lucene_index;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.GeneratedMessageV3;
import com.google.protobuf.Parser;

public class MapPersistentConfig<KeyClass, ValueClass, ProtoClass extends GeneratedMessageV3> extends BasePersistentConfig<ProtoClass> {
    private final AtomicReference<Map<KeyClass, ValueClass>> mappingRef = new AtomicReference<>();

    public MapPersistentConfig(YtLuceneIndexParams params, String name, Supplier<ProtoClass.Builder> builderSupplier,
                               Function<ProtoClass, KeyClass> keyGetter,
                               Function<ProtoClass, ValueClass> valueGetter) {
        super(params, name, builderSupplier);
        setIndexUpdateHandler(() -> {
            Map<KeyClass, ValueClass> newMapping = new HashMap<>();
            Parser<ProtoClass> parser = (Parser<ProtoClass>) builderSupplier.get().buildPartial().getParserForType();
            forEachProtoRecord(parser, proto -> {
                newMapping.put(keyGetter.apply(proto), valueGetter.apply(proto));
            });
            log.info("Got new mapping, size {}", newMapping.size());
            mappingRef.set(ImmutableMap.copyOf(newMapping));
        });
    }

    public ValueClass getByKey(KeyClass key) {
        return getAll().get(key);
    }

    public boolean containsKey(KeyClass key) {
        return getAll().containsKey(key);
    }

    public Map<KeyClass, ValueClass> getAll() {
        Map<KeyClass, ValueClass> mapping = mappingRef.get();
        if (mapping == null) {
            throw new IllegalStateException("PersistentConfig '" + name + "' is not ready");
        }
        return mapping;
    }
}
