package ru.yandex.travel.yt_lucene_index;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.protobuf.GeneratedMessageV3;
import com.google.protobuf.Parser;

public class MultiMapPersistentConfig<KeyClass, ValueClass, ProtoClass extends GeneratedMessageV3> extends BasePersistentConfig<ProtoClass> {
    private final AtomicReference<Map<KeyClass, Collection<ValueClass>>> mappingRef = new AtomicReference<>();

    public MultiMapPersistentConfig(YtLuceneIndexParams params, String name, Supplier<ProtoClass.Builder> builderSupplier,
                                    Function<ProtoClass, KeyClass> keyGetter,
                                    Function<ProtoClass, ValueClass> valueGetter,
                                    Function<Collection<ValueClass>, Collection<ValueClass>> valueListModifier) {
        super(params, name, builderSupplier);
        setIndexUpdateHandler(() -> {
            Multimap<KeyClass, ValueClass> newMapping = HashMultimap.create();
            Parser<ProtoClass> parser = (Parser<ProtoClass>) builderSupplier.get().buildPartial().getParserForType();
            forEachProtoRecord(parser, proto -> {
                newMapping.put(keyGetter.apply(proto), valueGetter.apply(proto));
            });
            Map<KeyClass, Collection<ValueClass>> newModifiedMapping = new HashMap<>();
            for (KeyClass key: newMapping.keySet()) {
                var values = newMapping.get(key);
                newModifiedMapping.put(key, valueListModifier.apply(values));
            }
            log.info("Got new mapping, size {}", newModifiedMapping.size());
            mappingRef.set(ImmutableMap.copyOf(newModifiedMapping));
        });
    }

    public Collection<ValueClass> getByKey(KeyClass key) {
        return getAll().get(key);
    }

    public boolean containsKey(KeyClass key) {
        return getAll().containsKey(key);
    }

    public Map<KeyClass, Collection<ValueClass>> getAll() {
        var mapping = mappingRef.get();
        if (mapping == null) {
            throw new IllegalStateException("PersistentConfig '" + name + "' is not ready");
        }
        return mapping;
    }
}
