package ru.yandex.travel.yt_lucene_index;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import NYT.Extension;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;

import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

class ProtoFiller {
    public static void fillByYtNode(YTreeMapNode node, Message.Builder proto, String namePrefix) {
        Descriptors.Descriptor d = proto.getDescriptorForType();

        for (Descriptors.FieldDescriptor fd : d.getFields()) {
            String fullName = namePrefix;
            if (!fullName.isEmpty()) {
                fullName += "_";
            }
            fullName += getColumnName(fd);
            Optional<YTreeNode> subNodeOpt = node.get(fullName);
            if (subNodeOpt.isEmpty()) {
                continue;
            }
            YTreeNode subNode = subNodeOpt.get();
            if (subNode.isEntityNode()) { // null values are empty entity nodes
                continue;
            }

            if (fd.isRepeated()) {
                if (subNode.isListNode()) {
                    fillByListNode(subNode, proto, fd, fullName);
                } else {
                    fillByYtNode(subNode.mapNode(), proto, fullName);
                }

                continue;
            }

            switch (fd.getType()) {
                case INT32:
                    proto.setField(fd, subNode.intValue());
                    break;
                case INT64:
                    proto.setField(fd, subNode.longValue());
                    break;
                case UINT32:
                    proto.setField(fd, subNode.intValue()); // protobuf in java stores uint32 as int
                    break;
                case UINT64:
                    proto.setField(fd, subNode.longValue());  // protobuf in java stores uint64 as long
                    break;
                case DOUBLE:
                    proto.setField(fd, subNode.doubleValue());
                    break;
                case FLOAT:
                    proto.setField(fd, subNode.floatValue());
                    break;
                case BOOL:
                    proto.setField(fd, subNode.boolValue());
                    break;
                case ENUM:
                    if (subNode.isStringNode()) {
                        Descriptors.EnumValueDescriptor ev = fd.getEnumType().findValueByName(subNode.stringValue());
                        if (ev != null) {
                            proto.setField(fd, ev);
                        } else {
                            throw new RuntimeException(String.format("Unknown string enum value %s for field '%s'",
                                    subNode.stringValue(), fullName));
                        }
                    } else if (subNode.isIntegerNode()) {
                        Descriptors.EnumValueDescriptor ev = fd.getEnumType().findValueByNumber(subNode.intValue());
                        if (ev != null) {
                            proto.setField(fd, ev);
                        } else {
                            throw new RuntimeException(String.format("Unknown string integer value %d for field '%s'"
                                    , subNode.intValue(), fullName));
                        }
                    }
                    break;
                case STRING:
                    proto.setField(fd, subNode.stringValue());
                    break;
                case MESSAGE:
                    fillByYtNode(subNode.mapNode(), proto.getFieldBuilder(fd), fullName);
                    break;
            }
        }
    }

    private static void fillByListNode(YTreeNode node, Message.Builder proto, Descriptors.FieldDescriptor fd,
                                       String fullName) {
        if (!node.isListNode()) {
            throw new RuntimeException("Repeated fields are not supported for non-list node");
        }

        for (var subNode : node.asList()) {
            switch (fd.getType()) {
                case INT32:
                case UINT32: // protobuf in java stores uint32 as int
                    proto.addRepeatedField(fd, subNode.intValue());
                    break;
                case INT64:
                case UINT64: // protobuf in java stores uint64 as long
                    proto.addRepeatedField(fd, subNode.longValue());
                    break;
                case DOUBLE:
                    proto.addRepeatedField(fd, subNode.doubleValue());
                    break;
                case FLOAT:
                    proto.addRepeatedField(fd, subNode.floatValue());
                    break;
                case BOOL:
                    proto.addRepeatedField(fd, subNode.boolValue());
                    break;
                case ENUM:
                    if (subNode.isStringNode()) {
                        Descriptors.EnumValueDescriptor ev = fd.getEnumType().findValueByName(subNode.stringValue());
                        if (ev != null) {
                            proto.addRepeatedField(fd, ev);
                        } else {
                            throw new RuntimeException(String.format("Unknown string enum value %s for field '%s'",
                                    subNode.stringValue(), fullName));
                        }
                    } else if (subNode.isIntegerNode()) {
                        Descriptors.EnumValueDescriptor ev = fd.getEnumType().findValueByNumber(subNode.intValue());
                        if (ev != null) {
                            proto.addRepeatedField(fd, ev);
                        } else {
                            throw new RuntimeException(String.format("Unknown string integer value %d for field '%s'"
                                    , subNode.intValue(), fullName));
                        }
                    }
                    break;
                case STRING:
                    proto.addRepeatedField(fd, subNode.stringValue());
                    break;
                case MESSAGE:
                    throw new RuntimeException(String.format("Repeated messages are not supported for field '%s'",
                            fullName));
            }
        }
    }

    private static void fillByMapNode(YTreeNode node, Message.Builder proto, Descriptors.FieldDescriptor fd) {
        if (!node.isMapNode()) {
            throw new RuntimeException("Repeated fields are not supported for current yt node");
        }

        proto.addRepeatedField(fd, getMapFromNode(node));
    }

    private static Map<String, Object> getMapFromNode(YTreeNode node) {
        if (node == null) {
            return null;
        }

        if (node.isListNode()) {
            throw new RuntimeException("List node can not be convert to map node");
        }

        Map<String, Object> result = new HashMap<>();

        for (var subNodeName : node.asMap().keySet()) {
            if (!node.asMap().containsKey(subNodeName)) {
                continue;
            }

            if (node.asMap().get(subNodeName).isListNode()) {
                result.put(subNodeName, getListFromNode(node.asMap().get(subNodeName)));
                continue;
            }

            if (node.asMap().get(subNodeName).isMapNode()) {
                result.put(subNodeName, getMapFromNode(node.asMap().get(subNodeName)));
                continue;
            }

            result.put(subNodeName, node.asMap().get(subNodeName));
        }

        return result;
    }

    private static List<Object> getListFromNode(YTreeNode node) {
        List<Object> result = new ArrayList<>();

        for (var item : node.asList()) {
            result.add(getMapFromNode(item));
        }

        return result;
    }

    private static String getColumnName(Descriptors.FieldDescriptor fieldDescriptor) {
        if (fieldDescriptor.getOptions().hasExtension(Extension.keyColumnName)) {
            return fieldDescriptor.getOptions().getExtension(Extension.keyColumnName);
        } else if (fieldDescriptor.getOptions().hasExtension(Extension.columnName)) {
            return fieldDescriptor.getOptions().getExtension(Extension.columnName);
        }
        return fieldDescriptor.getName();
    }
}
