# coding=utf-8
"""Контейнеры для информации о репликах."""
import logging  # noqa
import random
import six
import typing  # noqa

if six.PY3:
    from urllib.parse import urlencode
else:
    from urllib import urlencode

from travel.library.python.avia_mdb_replica_info.avia_mdb_replica_info.ping import ping_hosts

INF = float('inf')

_dc_to_order = {
    'man': ['man', 'iva', 'myt', 'vla', 'sas'],
    'iva': ['iva', 'myt', 'vla', 'sas', 'man'],
    'myt': ['myt', 'iva', 'vla', 'sas', 'man'],
    'sas': ['sas', 'vla', 'myt', 'iva', 'man'],
    'vla': ['vla', 'sas', 'myt', 'iva', 'man'],
}


class Replica(object):
    """Информация о реплике.

    :var hostname: имя хоста
    :var dc: идентификатор ДЦ, в котором находится хост
    :var is_master: является ли этот хост мастером
    :var raw_data: "сырые" данные о реплике, полученные из API Mdb
    """

    def __init__(self, hostname, dc, is_master=False, raw_data=None):
        # type: (str, str, bool, typing.Any) -> None
        self.hostname = hostname  # type: str
        self.dc = dc
        self.is_master = is_master
        self.raw_data = raw_data

    def __repr__(self):
        return '<%s hostname=%r dc=%r is_master=%r>' % (
            self.__class__.__name__,
            self.hostname,
            self.dc,
            self.is_master,
        )


class ClusterInfo(object):
    """Информация о инстансах кластера cluster_id.

    :var cluster_id: идентификатор кластера
    :var instances: все инстансы, включая мастер
    :var master: мастер
    :var replicas: только реплики
    """

    def __init__(self, cluster_id, instances):
        self.cluster_id = cluster_id  # type: str
        self.instances = instances  # type: typing.List[Replica]
        self.master = None  # type: typing.Optional[Replica]
        self.replicas = []  # type: typing.List[Replica]
        for replica in self.instances:
            if replica.is_master:
                self.master = replica
            else:
                self.replicas.append(replica)

    def __repr__(self):
        return '<%s instances=%r>' % (
            self.__class__.__name__,
            self.instances,
        )

    def _sort_replicas_by_ping(self):
        """Отсортировать реплики по времени пинга.

        ВАЖНО! Не работает из под supervisord и crond в Qloud.
        """
        ping_by_host = ping_hosts([replica.hostname for replica in self.instances])
        self.instances.sort(key=lambda replica: ping_by_host.get(replica.hostname, INF))

    def sort_replicas_by_dc(self, current_dc, logger=None):
        # type: (str, typing.Optional[logging.Logger]) -> None
        """Отсортировать реплики в заранее заданном порядке.

        :param current_dc: текущий ДЦ
        :param logger: логгер
        """
        logger = logger or logging.getLogger(__name__)
        if current_dc in _dc_to_order:
            order_by_dc = {
                dc: i
                for i, dc in enumerate(_dc_to_order[current_dc])
            }
            random.shuffle(self.instances)  # В одном ДЦ может быть более одной реплики, делаем случайный порядок
            self.instances.sort(key=lambda replica: (order_by_dc.get(replica.dc, INF), replica.is_master))
        else:
            logger.warning('Unknown DC %s', current_dc)

    def sort_replicas(self, current_dc, logger=None):
        # type: (str, typing.Optional[logging.Logger]) -> None
        """Отсортировать реплики.

        Сначала пытаемся отсортировать реплики по пингу, если не удалось, сортируем в заранее заданном порядке.

        :param current_dc: текущий ДЦ
        :param logger: логгер
        """
        logger = logger or logging.getLogger(__name__)
        try:
            self._sort_replicas_by_ping()
        except Exception:
            logger.exception('Cannot sort replicas by ping, falling back to hardcoded order')
            self.sort_replicas_by_dc(current_dc, logger)

    def psycopg2_connection_string(self, user, password, database, port=6432, **params):
        params['port'] = port
        return 'postgresql+psycopg2://{user}:{password}@{hosts}/{database}?{query_string}'.format(
            user=user,
            password=password,
            database=database,
            hosts=','.join(instance.hostname for instance in self.instances),
            query_string=urlencode(params),
        )
