# coding=utf-8
"""Получение информации о репликах из API MDB."""
import datetime
import fcntl
import json
import logging
import os
import time
import typing  # noqa

from dateutil.parser import isoparse
import requests
import requests.auth
import retrying
import six

from travel.library.python.avia_mdb_replica_info.avia_mdb_replica_info import containers, const


def _makedirs_exist_ok(path):
    if six.PY3:
        os.makedirs(path, exist_ok=True)
    else:
        try:
            os.makedirs(path)
        except os.error:
            if not os.path.isdir(path):
                raise


class MdbAPI(object):
    """Класс для получения информации о репликах из API MDB."""

    WAIT_FIXED_MS = 2000
    MAX_ATTEMPT_NUMBER = 4

    def __init__(
        self,
        api_base_url,
        oauth_token,
        cache_folder=const.DEFAULT_CACHE_FOLDER,
        iam_token_url=const.IAM_TOKEN_DEAFULT_URL,
        logger=None,
    ):
        # type: (str, str, str, str, typing.Optional[logging.Logger]) -> None
        """Создать экземпляр класса для получения информации о репликах из API Mdb.

        :param api_base_url: базовый url API Mdb
        :param oauth_token: OAuth токен
        :param
        :param iam_token_url: url для получения IAM токена
        :param logger: логгер
        """
        self.logger = logger or logging.getLogger(__name__)

        self._api_base_url = api_base_url
        self._session = requests.Session()

        self._oauth_token = oauth_token
        self._iam_token_url = iam_token_url
        self._iam_token_valid_till = 0

        self._default_cluster_info = {}

        self._cache_folder = cache_folder
        _makedirs_exist_ok(self._cache_folder)

    def _parse_response(self, cluster_id, response):
        # type: (str, typing.Dict) -> containers.ClusterInfo
        """Распарсить ответ MDB API.

        :param cluster_id: идентификатор кластера в MDB
        :param response: ответ MDB API
        :return: информация о кластере
        """
        self.logger.info('Parsing Mdb API response')
        try:
            return containers.ClusterInfo(cluster_id, [
                containers.Replica(
                    hostname=host['name'],
                    dc=host['zoneId'],
                    is_master=(host.get('role') == 'MASTER'),
                    raw_data=host,
                )
                for host in response['hosts']
            ])
        except Exception:
            self.logger.exception('Failed to parse response from MDB API: %s', response)
            raise

    @staticmethod
    def _save_json_file(filename, content):
        with open(filename, 'w') as cache_file:
            fcntl.lockf(cache_file, fcntl.LOCK_EX)
            json.dump(content, cache_file, indent=4)

    @staticmethod
    def _load_json_file(filename):
        with open(filename, 'r') as cache_file:
            fcntl.lockf(cache_file, fcntl.LOCK_SH)
            return json.load(cache_file)

    def _save_response(self, cluster_id, response):
        # type: (str, typing.Dict) -> None
        """Сохранить ответ API MDB.

        :param cluster_id: идентификатор кластера в MDB
        :param response: ответ API в виде распаршенного json
        """
        filename = os.path.join(self._cache_folder, '{}.json'.format(cluster_id))
        self._save_json_file(filename, response)

    def _load_cached_response(self, cluster_id):
        """Загрузить кэшированный ответ MDB API.

        :param cluster_id: идентификатор кластера в MDB
        :return: ответ API в виде распаршенного json
        """
        filename = os.path.join(self._cache_folder, '{}.json'.format(cluster_id))
        return self._load_json_file(filename)

    def _save_iam_token(self, response):
        self._save_json_file(os.path.join(self._cache_folder, 'iam_token.json'), response)

    def _load_iam_token(self):
        return self._load_json_file(os.path.join(self._cache_folder, 'iam_token.json'))

    def _request_iam_token(self):
        return self._session.post(
            url=self._iam_token_url,
            json={
                'yandexPassportOauthToken': self._oauth_token,
            }
        )

    def _update_iam_token(self):
        if self._iam_token_valid_till < time.time() - 10:
            try:
                response = self._request_iam_token()
                response_json = response.json()
                self._save_iam_token(response_json)
            except Exception:
                self.logger.exception('Unable to get IAM token')
                response = None
                try:
                    response_json = self._load_iam_token()
                except Exception:
                    self.logger.exception('Unable to load IAM token from cache')
                    response_json = None

            if response_json is not None and 'iamToken' in response_json:
                self._session.headers.update({
                    'Authorization': 'Bearer {}'.format(response_json['iamToken']),
                })

                if response is not None:
                    server_time = datetime.datetime.strptime(
                        response.headers['date'],
                        '%a, %d %b %Y %H:%M:%S GMT',
                    )
                    valid_till = isoparse(response_json['expiresAt']).replace(tzinfo=None)
                    self._iam_token_valid_till = time.time() + (valid_till - server_time).total_seconds()
            else:
                self.logger.error('Unable to get IAM token from response json: %s', response_json)

    def _request_cluster_info(self, cluster_id):
        # type: (str) -> typing.Dict
        """Получить информацию о репликах.

        :param cluster_id: идентификатор кластера
        :return: информация о репликах
        """
        url = '{}/clusters/{}/hosts'.format(self._api_base_url, cluster_id)

        @retrying.retry(
            wait_fixed=self.WAIT_FIXED_MS,
            stop_max_attempt_number=self.MAX_ATTEMPT_NUMBER,
        )
        def _get_json_response():
            self.logger.info('Trying to update IAM token')
            self._update_iam_token()
            self.logger.info('Making request to MDB API')
            r = self._session.get(url)
            self.logger.info('Got response from MDB API')
            return r.json()

        return _get_json_response()

    def add_default_cluster_info(self, cluster_id, master_hostname, hostnames):
        # type: (str, str, typing.List[str]) -> None
        """Добавить дефолтную информацию

        :param cluster_id: идентификатор кластера
        :param master_hostname: хост мастера
        :param hostnames: хосты реплик
        """
        if master_hostname not in hostnames:
            hostnames = hostnames + [master_hostname]

        cluster_info = containers.ClusterInfo(cluster_id, instances=[
            containers.Replica(
                hostname=hostname,
                dc=hostname[:3],
                is_master=(hostname == master_hostname),
            )
            for hostname in hostnames
        ])
        self._default_cluster_info[cluster_id] = cluster_info

    def get_default_cluster_info(self, cluster_id):
        return self._default_cluster_info.get(cluster_id)

    def get_cluster_info(self, cluster_id):
        # type: (str) -> containers.ClusterInfo
        """Получить информацию о кластере в MDB.

        :param cluster_id: идентификатор кластера
        :return: информация о кластере в MDB
        """
        try:
            response = self._request_cluster_info(cluster_id)
            cluster_info = self._parse_response(cluster_id, response)
            self._save_response(cluster_id, response)
        except Exception:
            self.logger.exception('Failed to query info about cluster %s from MDB API', cluster_id)
            try:
                response = self._load_cached_response(cluster_id)
                cluster_info = self._parse_response(cluster_id, response)
            except Exception:
                self.logger.exception('Failed to load cache for cluster %s', cluster_id)
                cluster_info = self.get_default_cluster_info(cluster_id)

        if cluster_info is None:
            raise Exception('Failed to get info about cluster %s' % cluster_id)

        return cluster_info
