# coding=utf-8
"""Функции для измерения времени пинга до хостов."""
import os
import re
import subprocess
import typing  # noqa

from travel.library.python.avia_mdb_replica_info.avia_mdb_replica_info import const

_ping_regex = re.compile(r'(\d+\.\d+)/(\d+\.\d+)/(\d+\.\d+)/(\d+\.\d+)')


def _get_avg(ping_stdout):
    # type: (str) -> float
    """Получить среднее значение пинга в миллисекундах из результата исполнения команды ping.

    :param ping_stdout: stdout команды ping
    :return: среднее значение пинга в миллисекундах
    """
    match = _ping_regex.search(ping_stdout)
    return float(match.group(2))


def create_ping_command(host, count, timeout, interval):
    cmd = [
        'ping6',
        '-c', str(count),
        '-i', str(interval),
    ]

    # ping6 -w on BSD systems like MacOS does different thing than on Linux.
    # There is no way to set timeout in BSD ping6.
    is_linux = os.uname()[0].lower() == 'linux'
    if is_linux:
        cmd += ['-w', str(timeout)]

    cmd += [host]
    return cmd


def ping_hosts(
    hosts,
    default=const.INF,
    count=const.PING_DEFAULT_COUNT,
    timeout=const.PING_DEFAULT_TIMEOUT,
    interval=const.PING_DEFAULT_INTERVAL,
):
    # type: (typing.List[str], float, int, float, float) -> typing.Dict[str, float]
    """Измерить среднее время ответа для хостов из списка.

    :param hosts: список хостов
    :param default: дефолтное значение, которое возвращается, если не удалось получить ответ от хоста
    :param count: количество пакетов
    :param timeout: максимальное время работы команды ping
    :param interval: интервал между пакетами
    :return: словарь, в котором ключи - хосты, а значения - среднее время ответа в миллисекундах
    """
    pings = {}
    for host in hosts:
        pings[host] = subprocess.Popen(
            create_ping_command(host, count, timeout, interval),
            stdout=subprocess.PIPE, stderr=subprocess.PIPE
        )
    result = {
        host: default
        for host in hosts
    }
    for host, ping in pings.items():
        out, error = ping.communicate()
        if error:
            raise Exception('Ping error: {}'.format(error))
        avg = _get_avg(out)
        result[host] = avg
    return result
