# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import attr
from attr import attrs, attrib
from typing import AnyStr, Callable, Iterable, Optional, Union

import requests
from pybreaker import CircuitBreaker, CircuitBreakerListener, CircuitBreakerStorage

from travel.library.python.base_http_client.errors import InvalidConfigException


@attrs
class CircuitBreakerConfig(object):
    fail_max = attrib(default=None, type=Optional[int])
    reset_timeout = attrib(default=None, type=Optional[int])
    exclude = attrib(default=None, type=Optional[Iterable[Union[Callable, Exception]]])
    listeners = attrib(default=None, type=Optional[Iterable[CircuitBreakerListener]])
    state_storage = attrib(default=None, type=Optional[CircuitBreakerStorage])
    name = attrib(default=None, type=Optional[AnyStr])

    def __attrs_post_init__(self):
        self.set_excluded_status_codes()

        if not self.is_valid:
            raise InvalidConfigException('Invalid CircuitBreakerConfig')

        self._configured_circuit_breaker = None

    def get_circuit_breaker(self):
        # type: () -> CircuitBreaker

        if not self._configured_circuit_breaker:
            self._configured_circuit_breaker = CircuitBreaker(**attr.asdict(self))
        return self._configured_circuit_breaker

    @property
    def is_valid(self):
        # type: () -> bool

        return not (self.fail_max is None or self.reset_timeout is None)

    def evolve(self, **mutated_params):
        return attr.evolve(self, **mutated_params)

    def set_excluded_status_codes(self, status_codes=(400, 401, 403, 404)):
        """Задать коды ответа, для которых не отрабатывает breaker"""
        self.exclude = [
            lambda e:
            type(e) is requests.HTTPError and e.response.status_code in status_codes
        ]
        return self
