# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from json import dumps as json_dumps
from typing import Any, AnyStr, Dict, Optional, Set
from six.moves.urllib.parse import urljoin

import requests
from pybreaker import CircuitBreakerError
from requests import Session
from requests.adapters import HTTPAdapter


from travel.library.python.tracing.instrumentation import traced_function
from travel.library.python.base_http_client.client_logger_context import client_logger_context
from travel.library.python.base_http_client.scope_configurator import ScopeConfigurator
from travel.library.python.base_http_client.types import (
    ClientNameType, CircuitBreakerConfigType, CustomHeadersCreatorsType, DisableType, HeadersType,
    RetryConfigType, RequestResultType, TimeoutType
)

log = logging.getLogger(__name__)


class BaseHttpClient(object):
    HTTP_CLIENT_NAME = None                 # type: ClientNameType
    DISABLE_TRACING = False                 # type: DisableType
    TIMEOUT = None                          # type: TimeoutType
    DISABLE_TIMEOUT = False                 # type: DisableType

    RETRY_CONFIG = None                     # type: RetryConfigType
    DISABLE_RETRY_CONFIG = False            # type: DisableType

    CIRCUIT_BREAKER_CONFIG = None           # type: CircuitBreakerConfigType
    DISABLE_CIRCUIT_BREAKER_CONFIG = False  # type: DisableType

    CUSTOM_HEADERS_CREATORS = ()            # type: CustomHeadersCreatorsType
    HEADERS = None                          # type: HeadersType

    MASKED_PARAMS = set()                   # type: Optional[Set]
    MASK_FOR_PARAMS_LOGGING = '*****'       # type: str

    def __init__(
        self,
        host,                                  # type: AnyStr
        timeout=None,                          # type: TimeoutType
        retry_config=None,                     # type: RetryConfigType
        circuit_breaker_config=None,           # type: CircuitBreakerConfigType
        custom_headers_creators=(),            # type: CustomHeadersCreatorsType
        headers=None,                          # type: HeadersType
        disable_timeout=None,                  # type: DisableType
        disable_retry_config=None,             # type: DisableType
        disable_circuit_breaker_config=None,   # type: DisableType
        disable_tracing=None,                  # type: DisableType
        masked_params=None                     # type: Optional[Set]
    ):
        client_scope = ScopeConfigurator(
            retry_config=self.RETRY_CONFIG,
            circuit_breaker_config=self.CIRCUIT_BREAKER_CONFIG,
            disable_timeout=self.DISABLE_TIMEOUT,
            disable_retry_config=self.DISABLE_RETRY_CONFIG,
            disable_circuit_breaker_config=self.DISABLE_CIRCUIT_BREAKER_CONFIG,
            disable_tracing=self.DISABLE_TRACING,
            timeout=self.TIMEOUT,
            custom_headers_creators=self.CUSTOM_HEADERS_CREATORS,
            headers=self.HEADERS,
            masked_params=self.MASKED_PARAMS
        )
        self._instance_scope = client_scope.merge(
            ScopeConfigurator(
                retry_config=retry_config,
                circuit_breaker_config=circuit_breaker_config,
                disable_timeout=disable_timeout,
                disable_retry_config=disable_retry_config,
                disable_circuit_breaker_config=disable_circuit_breaker_config,
                disable_tracing=disable_tracing,
                timeout=timeout,
                custom_headers_creators=custom_headers_creators,
                headers=headers,
                masked_params=masked_params
            )
        )

        self._host = host

    @property
    def http_client_name(self):
        return self.HTTP_CLIENT_NAME or self.__class__.__name__

    def _prepare_session(self, method_scope):
        # type: (ScopeConfigurator) -> Session

        session = Session()

        retry = method_scope.get_retry()
        if retry is not None:
            adapter = HTTPAdapter(max_retries=retry)
            session.mount(self._host, adapter)

        headers = method_scope.get_headers()
        session.headers.update(headers)

        return session

    def _make_params_for_log(self, params, masked_params):
        # type: (Optional[Dict], Optional[Set]) -> str

        result_params = {}
        if params:
            if not masked_params:
                result_params = params
            else:
                for key, value in params.items():
                    if key in masked_params:
                        result_params[key] = self.MASK_FOR_PARAMS_LOGGING
                    else:
                        result_params[key] = value

        return json_dumps(result_params)

    def _call(
        self,
        session,                    # type: Session
        url,                        # type: AnyStr
        method,                     # type: AnyStr
        params=None,                # type: Optional[Dict]
        masked_params=None,         # type: Optional[Set]
        data=None,                  # type: Any
        json=None,                  # type: Optional[Dict]
        timeout=None,               # type: TimeoutType
        check_response_status=True  # bool
    ):  # type: (...) -> RequestResultType

        params_for_log = self._make_params_for_log(params, masked_params)
        log_msg_header = '[{http_client_name} ({method} {url} with params {params})]'.format(
            http_client_name=self.http_client_name, method=method, url=url, params=params_for_log
        )

        log.info('{log_msg_header} call started'.format(log_msg_header=log_msg_header))
        try:
            response = session.request(
                method,
                url,
                params=params,
                data=data,
                json=json,
                timeout=timeout
            )
        except (requests.ConnectionError, requests.Timeout) as ex:
            log.exception('{log_msg_header} request error: {ex}'.format(log_msg_header=log_msg_header, ex=ex))
            raise

        if check_response_status:
            try:
                response.raise_for_status()
            except requests.HTTPError as ex:
                log.exception(
                    '{log_msg_header} response bad status: {status_code}, content: {content}, time: {elapsed}'.format(
                        log_msg_header=log_msg_header, status_code=ex.response.status_code,
                        content=ex.response.content, elapsed=ex.response.elapsed
                    )
                )
                raise

        log.info('{log_msg_header} request done, response code: {status_code}, time: {elapsed}'.format(
            log_msg_header=log_msg_header, status_code=response.status_code, elapsed=response.elapsed
        ))
        return response

    def get_instance_retry_config(self):
        # type: () -> RetryConfigType
        return self._instance_scope.retry_config

    def get_instance_circuit_breaker_config(self):
        # type: () -> CircuitBreakerConfigType
        return self._instance_scope.circuit_breaker_config

    def make_request(
        self,
        method,                                # type: AnyStr
        url_path,                              # type: AnyStr
        params=None,                           # type: Optional[Dict]
        masked_params=None,                    # type: Optional[Set]
        data=None,                             # type: Any
        json=None,                             # type: Optional[Dict]
        check_response_status=True,            # type: bool
        timeout=None,                          # type: TimeoutType
        retry_config=None,                     # type: RetryConfigType
        circuit_breaker_config=None,           # type: CircuitBreakerConfigType,
        custom_headers_creators=(),            # type: CustomHeadersCreatorsType
        headers=None,                          # type: HeadersType
        disable_timeout=None,                  # type: DisableType
        disable_retry_config=None,             # type: DisableType
        disable_circuit_breaker_config=None,   # type: DisableType
        disable_tracing=None                   # type: DisableType
    ):  # type: (...) -> RequestResultType

        method_scope = self._instance_scope.merge(
            ScopeConfigurator(
                retry_config=retry_config,
                circuit_breaker_config=circuit_breaker_config,
                disable_timeout=disable_timeout,
                disable_retry_config=disable_retry_config,
                disable_circuit_breaker_config=disable_circuit_breaker_config,
                disable_tracing=disable_tracing,
                timeout=timeout,
                custom_headers_creators=custom_headers_creators,
                headers=headers,
                masked_params=masked_params
            )
        )

        _call = self._call

        if not method_scope.disable_tracing:
            _call = traced_function(_call)

        circuit_breaker = method_scope.get_circuit_breaker()
        if circuit_breaker is not None:
            _call = circuit_breaker(_call)

        url = urljoin(self._host, url_path)
        with client_logger_context(self.http_client_name, method=method, url=url):
            with self._prepare_session(method_scope) as session:
                try:
                    return _call(
                        session,
                        url=url,
                        method=method,
                        params=params,
                        masked_params=method_scope.masked_params,
                        data=data,
                        json=json,
                        timeout=method_scope.get_timeout(),
                        check_response_status=check_response_status
                    )
                except CircuitBreakerError as ex:
                    log.exception('CircuitBreakerError: {ex}'.format(ex=ex))
                    raise

    def get(self, url_path, params=None, masked_params=None, **kwargs):
        return self.make_request('GET', url_path, params=params, masked_params=masked_params, **kwargs)

    def post(self, url_path, data=None, json=None, **kwargs):
        return self.make_request('POST', url_path, data=data, json=json, **kwargs)

    def put(self, url_path, data=None, json=None, **kwargs):
        return self.make_request('PUT', url_path, data=data, json=json, **kwargs)
