# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import attr
from attr import attrs, attrib
from typing import AnyStr, Iterable, Optional

import requests
from urllib3.util.retry import Retry

from travel.library.python.base_http_client.errors import InvalidConfigException


@attrs
class RetryConfig(object):
    total = attrib(default=None, type=Optional[int])
    connect = attrib(default=None, type=Optional[int])
    read = attrib(default=None, type=Optional[int])
    redirect = attrib(default=None, type=Optional[int])
    status = attrib(default=None, type=Optional[int])
    method_whitelist = attrib(default=Retry.DEFAULT_METHOD_WHITELIST, type=Iterable[str])
    status_forcelist = attrib(default=None, type=Optional[Iterable[int]])
    backoff_factor = attrib(default=0, type=float)
    raise_on_redirect = attrib(default=True, type=bool)
    raise_on_status = attrib(default=True, type=bool)
    history = attrib(default=None, type=Optional[tuple])
    respect_retry_after_header = attrib(default=True, type=bool)
    remove_headers_on_redirect = attrib(default=Retry.DEFAULT_REDIRECT_HEADERS_BLACKLIST, type=Iterable[AnyStr])

    def __attrs_post_init__(self):
        if not self.status_forcelist:
            self.set_excluded_status_codes()

        if not self.is_valid:
            raise InvalidConfigException('Invalid RetryConfig')
        self._configured_retry = None

    def get_retry(self):
        # type: () -> Retry

        if not self._configured_retry:
            self._configured_retry = Retry(**attr.asdict(self))
        return self._configured_retry

    @property
    def is_valid(self):
        # type: () -> bool

        return not all(
            param is None for param in [
                self.total,
                self.connect,
                self.read,
                self.redirect,
                self.status,
            ]
        )

    def evolve(self, **mutated_params):
        return attr.evolve(self, **mutated_params)

    def set_excluded_status_codes(self, status_codes=(400, 401, 403, 404)):
        """Задать коды ответа, для которых не происходит retry"""

        # https://stackoverflow.com/a/51701293
        self.status_forcelist = [
            x for x in requests.status_codes._codes
            if x >= 400 and x not in status_codes
        ]
        return self
