# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from urllib3.util.retry import Retry

from typing import Any, Optional, Set
from pybreaker import CircuitBreaker

from travel.library.python.base_http_client.errors import WrongClientConfigurationException
from travel.library.python.base_http_client.types import (
    CircuitBreakerConfigType, CustomHeadersCreatorsType, DisableType, HeadersType, RetryConfigType, TimeoutType
)


def _merge_field(self_field, other_field):
    # type: (Any, Any) -> Any
    return self_field if other_field is None else other_field


def _merge_config(self_config, other_config, self_disable_config, other_disable_config, config_name):
    if other_config is not None:
        return False, other_config

    disable_config = _merge_field(self_disable_config, other_disable_config)
    if disable_config:
        return True, None

    if self_config is None:
        raise WrongClientConfigurationException("Can't init client without {}".format(config_name))

    return False, self_config


def _merge_dicts(*dicts):
    result = {}
    for d in dicts:
        result.update(d)
    return result


class ScopeConfigurator:
    def __init__(
        self,
        retry_config=None,                    # type: RetryConfigType
        circuit_breaker_config=None,          # type: CircuitBreakerConfigType
        disable_timeout=None,                 # type: DisableType
        disable_retry_config=None,            # type: DisableType
        disable_circuit_breaker_config=None,  # type: DisableType
        disable_tracing=None,                 # type: DisableType
        timeout=None,                         # type: TimeoutType
        custom_headers_creators=(),           # type: CustomHeadersCreatorsType
        headers=None,                         # type: HeadersType
        masked_params=None                    # type: Optional[Set]
    ):
        if retry_config is not None and disable_retry_config:
            raise WrongClientConfigurationException(
                "Can't init with disable_retry_config=True and retry_config"
            )

        if circuit_breaker_config is not None and disable_circuit_breaker_config:
            raise WrongClientConfigurationException(
                "Can't init with disable_circuit_breaker_config=True and circuit_breaker_config"
            )

        if timeout is not None and disable_timeout:
            raise WrongClientConfigurationException(
                "Can't init with disable_timeout=True and timeout"
            )

        self.retry_config = retry_config
        self.circuit_breaker_config = circuit_breaker_config
        self.disable_timeout = disable_timeout
        self.disable_retry_config = disable_retry_config
        self.disable_circuit_breaker_config = disable_circuit_breaker_config
        self.disable_tracing = disable_tracing
        self.timeout = timeout
        self.custom_headers_creators = list(custom_headers_creators)
        self.headers = headers or {}
        self.masked_params = masked_params or set()

    def merge(
        self,
        init_scope  # type: ScopeConfigurator
    ):              # type: (...) -> ScopeConfigurator

        disable_retry_config, retry_config = _merge_config(
            self.retry_config, init_scope.retry_config,
            self.disable_retry_config, init_scope.disable_retry_config,
            'RetryConfig'
        )

        disable_circuit_breaker_config, circuit_breaker_config = _merge_config(
            self.circuit_breaker_config, init_scope.circuit_breaker_config,
            self.disable_circuit_breaker_config, init_scope.disable_circuit_breaker_config,
            'CircuitBreakerConfig'
        )

        disable_tracing = _merge_field(self.disable_tracing, init_scope.disable_tracing)
        disable_timeout, timeout = _merge_config(
            self.timeout, init_scope.timeout,
            self.disable_timeout, init_scope.disable_timeout,
            'timeout'
        )

        custom_headers_creators = self.custom_headers_creators + init_scope.custom_headers_creators
        headers = _merge_dicts(self.headers, init_scope.headers)

        masked_params = self.masked_params | init_scope.masked_params

        return ScopeConfigurator(
            retry_config=retry_config,
            circuit_breaker_config=circuit_breaker_config,
            disable_timeout=disable_timeout,
            disable_retry_config=disable_retry_config,
            disable_circuit_breaker_config=disable_circuit_breaker_config,
            disable_tracing=disable_tracing,
            timeout=timeout,
            custom_headers_creators=custom_headers_creators,
            headers=headers,
            masked_params=masked_params
        )

    def get_retry(self):
        # type: () -> Optional[Retry]

        if self.disable_retry_config:
            return None
        if self.retry_config is None:
            raise WrongClientConfigurationException('No RetryConfig in this scope')
        return self.retry_config.get_retry()

    def get_circuit_breaker(self):
        # type: () -> Optional[CircuitBreaker]

        if self.disable_circuit_breaker_config:
            return None
        if self.circuit_breaker_config is None:
            raise WrongClientConfigurationException('No CircuitBreakerConfig in this scope')
        return self.circuit_breaker_config.get_circuit_breaker()

    def get_timeout(self):
        # type: () -> TimeoutType
        return None if self.disable_timeout else self.timeout

    def get_headers(self):
        # type: () -> HeadersType

        _headers = {}
        _headers.update(self.headers)

        for custom_headers_creator in self.custom_headers_creators:
            _headers.update(custom_headers_creator.get_headers())

        return _headers
