# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from logging.handlers import WatchedFileHandler
import os


class FileTreeHandler(logging.Handler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    """
    SUFFIX = '.log'

    def __init__(self, base_path):
        logging.Handler.__init__(self)
        self.base_path = base_path
        self.handlers = {}

    def get_handler(self, name):
        if name in self.handlers:
            return self.handlers[name]

        parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        dirname = os.path.dirname(path)
        if not os.path.exists(dirname):
            os.makedirs(dirname)

        handler = self.handlers[name] = WatchedFileHandler(path)
        handler.formatter = self.formatter

        return handler

    def flush(self):
        for handler in self.handlers.values():
            handler.flush()

    def emit(self, record):
        try:
            handler = self.get_handler(record.name)
            handler.emit(record)
        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)


class ArcadiaFileTreeHandler(FileTreeHandler):
    """
    Обработчик, создающий дерево из обрабатываемых логов
    с возможностью обрезать ненужную часть пути до файлов,
    например 'travel.avia.backend'
    """

    def __init__(self, base_path, remove_prefix=None):
        super(ArcadiaFileTreeHandler, self).__init__(base_path)
        self.remove_prefix = remove_prefix

    def get_handler(self, name):
        if name in self.handlers:
            return self.handlers[name]

        if self.remove_prefix and name.startswith(self.remove_prefix):
            cut_length = len(self.remove_prefix) if self.remove_prefix.endswith('.') else len(self.remove_prefix) + 1
            parts = name[cut_length:].split('.')
        else:
            parts = name.split('.')

        path = os.path.join(self.base_path, *parts) + self.SUFFIX

        dirname = os.path.dirname(path)
        if not os.path.exists(dirname):
            os.makedirs(dirname)

        handler = self.handlers[name] = WatchedFileHandler(path)
        handler.formatter = self.formatter

        return handler
