# -*- coding: utf-8 -*-

import logging
from datetime import date, datetime
from fractions import Fraction
from typing import Optional

from dateutil.relativedelta import relativedelta
import requests

from travel.avia.library.python.safe_lxml import fromstring
from travel.library.python.tools import get_size_in_units, with_retries


class CurrencyConverter(object):
    REQUEST_TIMEOUT = 10
    # Search for the currency ID here: https://a.yandex-team.ru/arc_vcs/stocks/stocks/catalog/quotes.json
    # Verify currency ID here: https://yandex.ru/news/quotes/70004.html
    CURRENCY_INDEX = {
        'AMD': 10021,
        'AZN': 10022,
        'BYN': 10040,
        'CHF': 10007,
        'CNY': 10018,
        'EUR': 23,
        'GBP': 24,
        'ILS': 70004,
        'KGS': 10017,
        'KRW': 10019,
        'KZT': 10009,
        'PLN': 10031,
        'TRY': 10011,
        'UAH': 10006,
        'USD': 1,
        'UZS': 10012,
    }

    def __init__(self, base_url: str, start_date: Optional[date] = None):
        self.base_url = base_url
        self.rates = {}
        self.start_date = start_date or date(2000, 1, 1)
        for currency, currency_descriptor in self.CURRENCY_INDEX.items():
            logging.info('Processing currency %s', currency)
            self.rates[currency] = self.get_currency_rates(currency_descriptor)

    def get_currency_rates(self, currency_id: int) -> dict[date, float]:
        prices = self.get_raw_rates(currency_id)
        scale = self.get_currency_scale(currency_id)
        price_by_day = {}
        previous_date = self.start_date
        today = date.today()
        current_date = today
        rate = 0.
        for milliseconds, rate in prices:
            rate = float(rate / scale)
            current_date = (datetime.utcfromtimestamp(milliseconds // 1000)).date()
            if current_date < self.start_date:
                continue
            if current_date < previous_date:
                raise ValueError('Unordered prices')
            if (current_date - previous_date).days > 1:
                price_by_day.update(self.get_date_range(previous_date + relativedelta(days=+1), current_date, rate))
            price_by_day[current_date] = rate
            previous_date = current_date
        if current_date < today:
            price_by_day.update(self.get_date_range(current_date, today + relativedelta(days=+1), rate))
        return price_by_day

    @staticmethod
    def get_date_range(date_from: date, date_to: date, rate: float) -> dict[date, float]:
        result = dict()
        current_date = date_from
        while current_date < date_to:
            result[current_date] = rate
            current_date += relativedelta(days=+1)
        return result

    @with_retries
    def get_raw_rates(self, currency_id: int) -> dict[int, float]:
        url = self.base_url + '/graph_{}.json'.format(currency_id)
        rsp = requests.get(url, timeout=self.REQUEST_TIMEOUT)
        logging.debug('Got response of size %s bytes (%d bytes)', get_size_in_units(len(rsp.content)), len(rsp.content))
        return rsp.json()['prices']

    def get_rate(self, code, dt):
        return self.rates.get(code, {}).get(dt)

    def get_currency_scale(self, currency_id: int) -> Fraction:
        currency_meta = self.get_raw_currency_meta(currency_id)
        return Fraction(
            int(currency_meta.get('numerator_scale', 1)),
            int(currency_meta.get('denominator_scale', 1)),
        )

    @with_retries
    def get_raw_currency_meta(self, currency_id: int) -> dict:
        url = self.base_url + '/{}.xml'.format(currency_id)
        rsp = requests.get(url, timeout=self.REQUEST_TIMEOUT)
        logging.debug('Got response of size %s bytes (%d bytes)', get_size_in_units(len(rsp.content)), len(rsp.content))
        return {e.tag: e.text for e in fromstring(rsp.content)}
