# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

"""
Возможность запускать дебаггер pydevd в работающем процессе.

https://github.com/JetBrains/intellij-community/tree/master/python/helpers/pydev


Дебаггер запускается при отправке сигнала SIGUSR1 или SIGUSR2.

SIGUSR1 - запускает дебаггер в главном потоке, всё останавливается (аккуратно, если такое сделать с gunicorn worker, то мастер убьет воркера по таймауту)
SIGUSR2 - запускает отдельный тред с дебаггером, процесс продолжает свою работу

Основная идея такого дебаггера - получить возможность смотреть память и выполнять произвольный код, не останавливая процесс.

Пример запуска:
- добавляем при старте приложения сигналы через register_signals_for_debugger (например, при старте воркера)
- запускаем Pycharm Remote Debugging на localhost:8667. Pycharm ожидает подключения дебаггера.
https://www.jetbrains.com/help/pycharm/remote-debugging-with-product.html#remote-debug-config
- делаем тоннель с порта 8666 удаленной машины, где крутится приложение, на наш локальный порт 8667
ssh -N -R 8666:localhost:8667 sas1-07171947eeca.qloud-c.yandex.net
- находим pid нашего процесса и посылаем ему сигнал о запуске дебаггера
kill -s SIGUSR2 4242  # 4242 это pid
- процесс запускает дебаггер, который коннектится на локальный порт 8666 и пробрасывается в наш pycharm на 8667 порт

"""

import os
import signal
from threading import Thread


from travel.library.python.dev.pydevd_inject import install_pydevd


def connect_to_remote_debugger(host=None, port=None, **settrace_kwargs):
    host = host or os.getenv('RASP_PYDEVD_REMOTE_HOST', '127.0.0.1')
    port = port or os.getenv('RASP_PYDEVD_REMOTE_PORT', 8666)

    install_pydevd.install_pydevd()
    import pydevd

    print('Connecting pydevd client to {}:{}'.format(host, port))
    kwargs = dict(stdoutToServer=True, stderrToServer=True)
    kwargs.update(settrace_kwargs)
    pydevd.settrace(host, port=port, **kwargs)
    print('Connection closed: pydevd client to {}:{}'.format(host, port))


def run_in_thread(*args, **kwargs):
    print('Starting pydevd client thread', args, kwargs)
    thread = Thread(target=connect_to_remote_debugger)
    thread.start()


def run_in_main_thread(*args, **kwargs):
    print('Starting pydevd client synchronously in main thread', args, kwargs)
    connect_to_remote_debugger()


def register_signals_for_debugger():
    signal.signal(signal.SIGUSR1, run_in_main_thread)
    signal.signal(signal.SIGUSR2, run_in_thread)
