import six
import struct
from io import BytesIO

UINT32 = struct.Struct('<I')


def _iload_messages(stream):
    while True:
        raw_size = stream.read(UINT32.size)
        if not raw_size:
            break
        size, = UINT32.unpack(raw_size)

        yield stream.read(size)


def iload_messages_from_string(content):
    for msg in _iload_messages(BytesIO(content)):
        yield msg


def iload_messages(path):
    with open(path, 'rb') as stream:
        for msg in _iload_messages(stream):
            yield msg


def dump_messages(stream, messages):
    for message in messages:
        stream.write(UINT32.pack(len(message)))
        stream.write(message)


class BaseRepository(object):
    _PB = None
    _default_index_field = "Id"

    def __init__(self, index_field=None):
        self._index_field = index_field or self._default_index_field
        self._storage = {}

    def _parse(self, proto):
        return self._PB.FromString(proto)

    def add(self, proto):
        parsed_proto = self._parse(proto)
        self.add_object(parsed_proto)

    def add_object(self, obj):
        self._storage[getattr(obj, self._index_field)] = obj

    def add_objects(self, objects):
        for obj in objects:
            self.add_object(obj)

    def get(self, id):
        return self._storage.get(id)

    def itervalues(self):
        return six.itervalues(self._storage)

    def load_from_file(self, path):
        for proto in iload_messages(path):
            self.add(proto)

    def load_from_string(self, content):
        for proto in iload_messages_from_string(content):
            self.add(proto)

    def dump_to_file(self, path):
        with open(path, 'wb') as stream:
            self.dump_to_stream(stream)

    def dump_to_stream(self, stream):
        dump_messages(stream, (self._PB.SerializeToString(obj) for obj in self.itervalues()))

    def size(self):
        return len(self._storage)

    @classmethod
    def get_proto_class(clazz):
        return clazz._PB


class BaseListRepository(BaseRepository):
    """
    Same as base repository, but uses list, not diсt as a storage
    """

    def __init__(self):
        BaseRepository().__init__(self)
        self._storage = []

    def add_object(self, obj):
        self._storage.append(obj)

    def add_objects(self, objects):
        self._storage.extend(objects)

    def get(self, pos):
        return self._storage[pos]

    def values(self):
        return self._storage

    def itervalues(self):
        return self._storage
