# coding: utf-8
from __future__ import unicode_literals, absolute_import, print_function, division

from travel.library.python.dicts.carrier_repository import CarrierRepository
from travel.library.python.dicts.country_repository import CountryRepository
from travel.library.python.dicts.currency_repository import CurrencyRepository
from travel.library.python.dicts.factories.carrier import TCarrierFactory
from travel.library.python.dicts.factories.country import TCountryFactory
from travel.library.python.dicts.factories.currency import TCurrencyFactory
from travel.library.python.dicts.factories.pointsynonym import TPointSynonymFactory
from travel.library.python.dicts.factories.region import TRegionFactory
from travel.library.python.dicts.factories.settlement import TSettlementFactory
from travel.library.python.dicts.factories.station import TStationFactory
from travel.library.python.dicts.factories.timezone import TTimeZoneFactory
from travel.library.python.dicts.pointsynonym_repository import PointSynonymRepository
from travel.library.python.dicts.region_repository import RegionRepository
from travel.library.python.dicts.settlement_repository import SettlementRepository
from travel.library.python.dicts.station_repository import StationRepository
from travel.library.python.dicts.timezone_repository import TimezoneRepository

MSK_TZ_ID = 33
EKB_TZ_ID = 44


class RaspRepositories(object):
    """
    Для тестов нужно иметь консистентные данные в репозиториях прото-объектов.
    Аналогией служит база данных для тестов. В БД заполнена структура и некоторые начальные данные.

    В этом классе создаем репозитории и делаем создание объектов в репозиториях более консистентным.
    Также создаем некоторые начальные данные.
    """
    def __init__(self):
        self.currency_repository = CurrencyRepository()

        self.tz_repository = TimezoneRepository()
        self.tz_repository.add_object(TTimeZoneFactory(Id=MSK_TZ_ID, Code='Europe/Moscow'))
        self.tz_repository.add_object(TTimeZoneFactory(Id=EKB_TZ_ID, Code='Asia/Yekaterinburg'))

        self.country_repository = CountryRepository()
        self.region_repository = RegionRepository()
        self.settlement_repository = SettlementRepository()
        self.station_repository = StationRepository()

        self.carrier_repository = CarrierRepository()
        self.pointsynonym_repository = PointSynonymRepository()

    def create_currency(self, **kwargs):
        currency = TCurrencyFactory(**kwargs)
        self.currency_repository.add_object(currency)
        return currency

    def create_timezone(self, **kwargs):
        tz = TTimeZoneFactory(**kwargs)
        self.tz_repository.add_object(tz)
        return tz

    def create_country(self, **kwargs):
        country = TCountryFactory(**kwargs)
        self.country_repository.add_object(country)
        return country

    def create_region(self, **kwargs):
        kwargs.setdefault('TimeZoneId', MSK_TZ_ID)
        region = TRegionFactory(**kwargs)
        self.region_repository.add_object(region)
        return region

    def create_settlement(self, **kwargs):
        kwargs.setdefault('TimeZoneId', MSK_TZ_ID)
        settlement = TSettlementFactory(**kwargs)
        self.settlement_repository.add_object(settlement)
        return settlement

    def create_station(self, **kwargs):
        kwargs.setdefault('TimeZoneId', MSK_TZ_ID)
        station = TStationFactory(**kwargs)
        self.station_repository.add_object(station)
        return station

    def create_carrier(self, **kwargs):
        carrier = TCarrierFactory(**kwargs)
        self.carrier_repository.add_object(carrier)
        return carrier

    def create_pointsynonym(self, **kwargs):
        pointsynonym = TPointSynonymFactory(**kwargs)
        self.pointsynonym_repository.add_object(pointsynonym)
        return pointsynonym
