# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import signal
import subprocess
import sys
import threading
import time
from StringIO import StringIO

from apscheduler.events import EVENT_SCHEDULER_STARTED
from apscheduler.schedulers.blocking import BlockingScheduler
from apscheduler.triggers.cron import CronTrigger
from apscheduler.triggers.interval import IntervalTrigger
from flask import Flask


log = logging.getLogger(__name__)


class ProcessError(Exception):
    pass


def check_wait_status(wait_status, command_args):
    """
    :param wait_status: waitpid status
    :return: True if exited successfully
    :rtype: bool
    """
    if os.WIFSIGNALED(wait_status):
        raise ProcessError('Process {} was killed with signal {}'.format(
            ' '.join(command_args),
            os.WTERMSIG(wait_status))
        )
    if os.WIFEXITED(wait_status):
        if os.WEXITSTATUS(wait_status) != 0:
            raise ProcessError('Process {} exited with status {}'.format(
                ' '.join(command_args),
                os.WEXITSTATUS(wait_status)
            ))
        else:
            log.info('Process successfully finished %s', ' '.join(command_args))
            return True
    return False


def run_job(*args, **kwargs):
    """
    Мы запускаем скрипты в отдельных процессах для того, чтобы не упереться в GIL при попытке выполнить,
    несколько CPU bound операций параллельно внутри threadpool.
    А также чтобы иметь возможность безболезненно убить джоб.
    """

    process_env = os.environ.copy()
    if kwargs.get('env'):
        process_env.update(kwargs.get('env'))

    timeout = kwargs.get('timeout', 3600)

    proc = subprocess.Popen(list(args), env=process_env)
    start = time.time()
    while True:
        pid, wait_status = os.waitpid(proc.pid, os.WNOHANG)
        if pid and check_wait_status(wait_status, args):
            break

        if time.time() - start < timeout:
            time.sleep(0.5)
        else:
            log.error('Process {} with pid {} exceeded timeout {} killing with SIGTERM'.format(' '.join(args), proc.pid, timeout))
            os.kill(proc.pid, signal.SIGTERM)
            pid, wait_status = os.waitpid(proc.pid, os.WNOHANG)
            if pid and check_wait_status(wait_status, args):
                raise ProcessError('Process {} with pid {} exceeded timeout {} and killed with SIGTERM'
                                   .format(' '.join(args), proc.pid, timeout))

            time.sleep(1)
            log.error('Process {} with pid {} exceeded timeout {} and hung killing with SIGKILL'.format(' '.join(args), proc.pid, timeout))
            os.kill(proc.pid, signal.SIGKILL)
            pid, wait_status = os.waitpid(proc.pid, os.WNOHANG)
            if pid:
                check_wait_status(wait_status, args)
            raise ProcessError('Process {} with pid {} exceeded timeout {} and killed with SIGKILL'
                               .format(' '.join(args), proc.pid, timeout))


class Scheduler(object):
    def __init__(self, launch_entrypoint, with_web_server=False):
        self.launch_entrypoint = launch_entrypoint
        self.with_web_server = with_web_server

        jobstores = {'default': {'type': 'memory'}}
        executors = {'default': {'type': 'threadpool', 'max_workers': 30}}
        job_defaults = {
            'coalesce': True,  # Игнорировать не отработанные запуски в прошлом
            'max_instances': 1  # Не запускать больше одного таска одного типа подряд
        }

        self.scheduler = BlockingScheduler()
        self.scheduler.configure(jobstores=jobstores, executors=executors, job_defaults=job_defaults, timezone='Europe/Moscow')
        self.scheduler.add_listener(self.start_handler, EVENT_SCHEDULER_STARTED)
        self.web_server = None

    def run(self):
        log.info('Starting scheduler')
        signal.signal(signal.SIGINT, self.shutdown_handler)

        self.scheduler.start()
        log.info('Scheduler Exited')

    def start_handler(self, event):
        if self.with_web_server:
            self.web_server = SchedulerWebServer(self)
            self.web_server.start()

    def shutdown_handler(self, *args, **kwargs):
        signal.signal(signal.SIGINT, signal.SIG_DFL)
        log.info('Shutting Down Scheduler')
        self.scheduler.shutdown()

    def add_jobs(self, jobs):
        for job_key, conf in jobs.items():
            schedule = conf['options'].get('schedule')
            if not schedule:
                continue

            log.info('Adding job to schedule: %s', job_key)

            if 'cron' in schedule:
                trigger = CronTrigger.from_crontab(schedule['cron'])
            elif 'interval' in schedule:
                trigger = IntervalTrigger(seconds=schedule['interval'])
            else:
                trigger = schedule['trigger']

            # запускаем наш же бинарник с переопределением точки входа
            job_args = [sys.executable, job_key] + schedule.get('args', [])
            job_env = {'Y_PYTHON_ENTRY_POINT': self.launch_entrypoint}

            if schedule.get('env'):
                job_env.update(schedule['env'])

            self.add_job(job_key, trigger, job_args=job_args, env=job_env, timeout=schedule.get('timeout'))

    def add_job(self, job_key, trigger, timeout=None, env=None, job_args=None):
        runner_kwargs = {}

        if timeout is not None:
            runner_kwargs['timeout'] = timeout

        if env is not None:
            runner_kwargs['env'] = env

        params = {
            'func': run_job,
            'args': job_args,
            'kwargs': runner_kwargs,
            'id': job_key,
            'name': job_key,
            'trigger': trigger,
        }

        return self.scheduler.add_job(**params)

    def print_jobs(self):
        result = StringIO()
        self.scheduler.print_jobs(out=result)
        result.seek(0)
        return result.read()


class SchedulerWebServer(object):
    def __init__(self, scheduler, host='::1', port=8888):
        self.scheduler = scheduler

        self.flask_app = Flask(__name__)
        self.flask_app.route('/ping')(self.handler_ping)
        self.flask_app.route('/list')(self.handler_list)

        self.server_thread = None
        self.host = host
        self.port = port

    def start(self):
        self.server_thread = threading.Thread(
            target=self.flask_app.run,
            kwargs=dict(host=self.host, port=self.port, load_dotenv=False)
        )
        self.server_thread.daemon = True
        self.server_thread.start()

    def handler_ping(self):
        return 'pong\n'

    def handler_list(self):
        return self.scheduler.print_jobs()
