# coding=utf-8
import logging
import os
from typing import List, Optional

import kikimr.public.sdk.python.persqueue.auth as auth
import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib
from travel.library.python.logbroker.consumer import LogbrokerConsumer

logger = logging.getLogger(__name__)


class LogbrokerClient(object):
    """Минимальный клиент для логброкера.
    with LogbrokerClient() as logbroker:
        ...
    """
    timeout = 15  # type: int

    def __init__(self, token='', endpoint='', timeout=None):
        """
        :param str token: Токен доступа.
            Если не передан, будет взят из переменной окружения LOGBROKER_TOKEN
        :param str endpoint: endpoint инсталляции.
            Если не передан, будет использован перенаправляющий записи в случае отказа дц
            logbroker.yandex.net
        :param Optional[int] timeout: Таймаут на операции.
            Если не передан, будет взято значение из одноимённого атрибута класса.
        """
        self._api = None  # type: Optional[pqlib.PQStreamingAPI]
        self.timeout = timeout or self.timeout
        self.endpoint = endpoint or 'logbroker.yandex.net'

        token = token or os.environ.get('LOGBROKER_TOKEN', '')

        self.credentials = auth.OAuthTokenCredentialsProvider(token.encode())

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.quit()

    def quit(self):
        api = self._api
        if not api:
            return
        try:
            api.stop()
        except:
            logger.exception('Failed to stop logbroker API for endpoint %s', self.endpoint)

    def _init(self):
        api = pqlib.PQStreamingAPI(self.endpoint, 2135)
        api_start_future = api.start()
        api_start_future.result(timeout=self.timeout)

        self._api = api
        return api

    @property
    def api(self):
        api = self._api

        if not api:
            api = self._init()

        return api

    def get_consumer(self, name, topics, json_decoder=None, silent=False, balance_partition_now=False):
        """Возвращает читателя для определённого раздела.
        :param name: Идентификатор, описывающий читателя.
        :param List[str] topics: Разделы, откуда требуется производить чтение.
        :param Union[bool, Optional[Type[json.JSONDecoder]]] json_decoder: Декодировщик json, используемый при чтении данных.
        :param bool silent: Флаг тихого режима. В тихом режиме исключения иницилизации
            читателя не поднимаются.
        :return LogbrokerConsumer:
        """
        return LogbrokerConsumer(
            client=self,
            name=name,
            topics=topics,
            json_decoder=json_decoder,
            silent=silent,
            balance_partition_now=False,
        )


DEFAULT_DC_ENDPOINTS = (
    'sas.logbroker.yandex.net',
    'vla.logbroker.yandex.net',
    'iva.logbroker.yandex.net',
)


class LogbrokerReader(object):
    """Читатель для многокластерных инсталляций логброкера.
    Позволяет установить соединение со всеми кластерами и читать данные из них,
    используя единый интерфейс.
    """

    def __init__(
            self, name, topics, json_decoder=None, token='', timeout=None, silent=True, endpoints=DEFAULT_DC_ENDPOINTS,
            balance_partition_now=False,
    ):
        """
        :param str name: Идентификатор, описывающий читателя.
        :param List[str] topics: Разделы, откуда требуется производить чтение.
        :param Union[bool, Optional[Type[json.JSONDecoder]]] json_decoder: Декодировщик json, используемый при чтении данных.
        :param str token: Токен доступа.
            Если не передан, будет взят из переменной окружения LOGBROKER_TOKEN
        :param Optional[int] timeout: Таймаут на операции.
            Если не передан, будет взято значение из одноимённого атрибута класса.
        :param bool silent: Флаг тихого режима. В тихом режиме исключения иницилизации
            читателя не поднимаются.
        :param List[str] endpoints: Список endpoint-ов для чтения
        :param bool balance_partition_now: Перебалансиировать партиции при создании сессии
        """
        clients = []
        consumers = []

        self._clients = clients  # type: List[LogbrokerClient]
        self._consumers = consumers  # type: List[LogbrokerConsumer]

        for endpoint in endpoints:
            client = LogbrokerClient(
                token=token,
                endpoint=endpoint,
                timeout=timeout,
            )
            consumer = client.get_consumer(
                name=name,
                topics=topics,
                json_decoder=json_decoder,
                silent=silent,
                balance_partition_now=balance_partition_now,
            )
            clients.append(client)
            consumers.append(consumer)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):

        for consumer in self._consumers:
            consumer.close()

        for client in self._clients:
            client.quit()

    def read(self, limit_batches=None, retry_on_timeout=False):
        """Генератор производит чтение сообщений.
        Последовательно из каждого кластера вычитываются все доступные
        сообщения (если не указан limit_batches).
        Если за указанное в timeout_read время не удаётся
        прочитать сообщение, предполагается, что сообщений нет.
        :param Optional[int] limit_batches: Максимальное количество пакетов сообщений для вычитки.
            Указание, заставит процесс вычики остановится.
            Внимание: в пришедшем пакете может быть больше одного сообщения.
        :return Generator[LogbrokerMessage, None, None]
        """
        for consumer in self._consumers:
            for message in consumer.read(limit_batches=limit_batches, retry_on_timeout=retry_on_timeout):
                yield message

    def read_batch(self, retry_on_timeout=False):
        """Генератор производит чтение пакета сообщений.
        Последовательно из каждого кластера вычитывается следующий пакета сообщений
        Если за указанное в timeout_read время не удаётся
        прочитать сообщение, предполагается, что сообщений нет.
        :return Generator[Generator[LogbrokerMessage, None, None], None, None]
        """
        for consumer in self._consumers:
            yield consumer.read(limit_batches=1, retry_on_timeout=retry_on_timeout)
