# coding=utf-8
import json
import logging
from concurrent.futures import TimeoutError
from typing import List, Optional, Type  # noqa
from collections import namedtuple

import kikimr.public.sdk.python.persqueue.grpc_pq_streaming_api as pqlib
import kikimr.public.sdk.python.persqueue.errors as errors

from travel.library.python.logbroker.exceptions import LogbrokerException

if False:
    from travel.library.python.logbroker.client import LogbrokerClient  # noqa

logger = logging.getLogger(__name__)

LogbrokerMessage = namedtuple('LogbrokerMessage', ['seq_num', 'source', 'topic', 'endpoint', 'data'])


class LogbrokerConsumer(object):
    """Читатель из логброкера.
    with LogbrokerClient() as logbroker:
        with logbroker.get_consumer(name='here/is/my/consumer', topic='some/there/here') as consumer:
            for message in consumer.read():
                ...
    """
    default_init_timeout = 10  # type: int
    default_read_timeout = 10  # type: int
    default_max_batch_size = 64  # type: int
    default_max_bytes_per_batch = 1024 * 1024 * 200  # type: int
    default_read_infly_count = 2  # type: int

    def __init__(self, client, name, topics, init_timeout=None, read_timeout=None, json_decoder=None, silent=False, max_batch_size=None,
                 max_bytes_per_batch=None, read_infly_count=None, balance_partition_now=False):
        """
        :param LogbrokerClient client: Экземпляр клиента.
        :param str name: Наименование (оно же путь) читателя, от имени которого будет производится чтение.
        :param List[str] topics: Наименования разделов (они же пути), из которых будет производиться чтение.
        :param Optional[int] init_timeout:  Таймаут на инициализацию читателя.
            Если не передан, будет взято значение из одноимённого атрибута класса.
        :param Optional[int] read_timeout:  Таймаут на операции чтения.
            Если не передан, будет взято значение из одноимённого атрибута класса.
        :param Union[bool, Optional[Type[json.JSONDecoder]]] json_decoder: Декодировщик json, используемый при чтении данных в виде словарей.
            Можно указать конкретный класс декодировщика, либо True для использования умолчательного.
            В случае невозможности докодирования вместо тела сообщения вернётся None.
        :param bool silent: Флаг тихого режима. В тихом режиме исключения иницилизации
            читателя не поднимаются.
        """
        self.init_timeout = init_timeout or self.default_init_timeout
        self.read_timeout = read_timeout or self.default_read_timeout
        self.client = client
        self.topics = topics
        self.name = name
        self.silent = silent
        self._max_batch_size = max_batch_size or self.default_max_batch_size
        self._max_bytes_per_batch = max_bytes_per_batch or self.default_max_bytes_per_batch
        self._read_infly_count = read_infly_count or self.default_read_infly_count
        self._balance_partition_now = balance_partition_now

        if json_decoder is True:
            json_decoder = json.JSONDecoder

        self.json_decoder = json_decoder
        self._consumer = None
        self._event_future = None

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()

    def close(self):
        consumer = self._consumer
        if not consumer:
            return
        try:
            consumer.reads_done()
            consumer.stop()
            self._event_future = None
            self._consumer = None
        except:
            logger.exception('Failed to close consumer for endpoint %s', self.client.endpoint)

    def init(self):
        # type: () -> Optional[LogbrokerConsumer]
        """Инициализирует читателя.
        В случае необходимости будет автоматически вызван в методе .read().
        """
        self._event_future = None
        client = self.client
        silent = self.silent
        err_prefix = 'Unable to initialize Logbroker consumer:'

        configurator = pqlib.ConsumerConfigurator(
            [topic.encode() for topic in self.topics],
            self.name.encode(),
            max_count=self._max_batch_size,
            max_size=self._max_bytes_per_batch,
            read_infly_count=self._read_infly_count,
            balance_partition_now=self._balance_partition_now,
        )

        def silent_fail(msg):
            self._consumer = None
            logger.error(msg)
            return silent

        try:
            consumer = client.api.create_consumer(configurator, credentials_provider=client.credentials)

        except TimeoutError:
            if silent_fail('{} timeout reached'.format(err_prefix)):
                return None
            raise

        start_future = consumer.start()
        result = start_future.result(timeout=self.init_timeout)

        if isinstance(result, errors.SessionFailureResult):
            description = result.description

            if description:
                error = description.error.description
                errcode = description.error.code

            else:
                error = str(result)
                errcode = '?'

            if silent_fail('{} {} [{}]'.format(err_prefix, error, errcode)):
                return None

            raise LogbrokerException('{} {}'.format(err_prefix, error), errcode)

        if not result.HasField('init'):
            error = '{} {}'.format(err_prefix, result)

            if silent_fail(error):
                return None

            raise LogbrokerException(error)

        self._consumer = consumer
        return consumer

    def read(self, limit_batches=None, retry_on_timeout=False):
        """Генератор производит чтение сообщений.
        Вычитываются все доступные сообщения (если не указан limit_batches).
        Если за указанное в self.read_timeout время не удаётся
        прочитать сообщение, предполагается, что сообщений нет.
        :param Optional[int] limit_batches: Максимальное количество пакетов сообщений для вычитки.
            Указание, заставит процесс вычики остановится.
            Внимание: в пришедшем пакете может быть больше одного сообщения.
        :param bool retry_on_timeout: Следует ли повторять попытки чтения,
            если был таймаут (может означать, что сообщений нет).
        :returns Generator[LogbrokerMessage, None, None]:
        """

        def init_consumer():
            """[Пере]инициализирует читателя.
            Есть вероятность неудачи инициализации, например из-за тайм-аута.
            """
            self.close()
            self.init()
            return self._consumer

        consumer = self._consumer
        if not consumer:
            consumer = init_consumer()
            if not consumer:
                return

        received_batches = 0

        if limit_batches is None:
            limit_batches = float('inf')

        while True:
            try:
                event = self._get_next_event()
            except TimeoutError:
                if retry_on_timeout:
                    continue
                return

            except errors.ActorTerminatedException:
                consumer = init_consumer()
                if not consumer:
                    return

                continue

            event_type = event.type

            if event_type == pqlib.ConsumerMessageType.MSG_DATA:
                batch_response = event.message.data
                cookie_latest = batch_response.cookie

                for batch in batch_response.message_batch:
                    for message in self._extract_messages_from_batch(batch):
                        yield message

                consumer.commit(cookie_latest)
                received_batches += 1

                if received_batches >= limit_batches:
                    return

            elif event_type == pqlib.ConsumerMessageType.MSG_ERROR:
                # Ошибка. Следует повторно инициализировать клиента.
                consumer = init_consumer()
                if not consumer:
                    return

            # Сюда попадут прочие (необрабатываемые нами) команды типа LOCK, RELEASE и MSG_COMMIT.

    def _get_next_event(self):
        """
        Если future полученная из вызова next_event затаймаутилась (в топике нет сообщений),
        то когда сообщения появятся, они придут именно в эту future,
        поэтому её нужно сохранить и пытаться получать из неё данные вызовом .result(timeout)
        """
        if self._event_future is None:
            self._event_future = self._consumer.next_event()

        event = self._event_future.result(timeout=self.read_timeout)
        self._event_future = None
        return event

    def _extract_messages_from_batch(self, batch):
        topic = _format_topic(batch.topic)

        for message in batch.message:
            meta = message.meta
            source = meta.source_id.decode()
            seq_num = meta.seq_no

            data = message.data

            if self.json_decoder:
                try:
                    data = json.loads(data, cls=self.json_decoder)

                except ValueError:
                    logger.warning('Invalid JSON at %s. Seq %d. Source: %s', topic, seq_num, source)
                    data = None

            yield LogbrokerMessage(
                seq_num=seq_num,
                source=source,
                topic=topic,
                endpoint=self.client.endpoint,
                data=data,
            )


def _format_topic(topic):
    # type: (str) -> str
    """
    Преобразуем строки типа 'rt3.myt--mdh@test--trash'
    в mdh/test/trash
    """
    _, _, topic = topic.replace('--', '/').replace('@', '/', 1).partition('/')
    return topic
