# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import argparse
import base64
import gzip
import io
import json
import sys

import codecs
import os
import ast
from collections import OrderedDict

import six

from travel.library.python.rasp_vault import common, qloud
from travel.library.python.rasp_vault.common import VaultSecret, VaultVersion

DOC = """
Генерирует выгрузку секретов из секретницы
"""


class CommandError(Exception):
    pass


class BadExpressionError(CommandError):
    pass


class ExtraSecretsNotFoundError(CommandError):
    pass


def get_vault_client(args):
    return common.get_vault_client()


def gen_versions(args):
    client = get_vault_client(args)
    if not client:
        return []

    versions = []
    page = 0
    extra_secrets = list(set(args.extra or []))
    while True:
        secret_list = client.list_secrets(page_size=50, page=page)
        page += 1
        if not secret_list:
            break
        for list_secret_item in secret_list:
            secret = VaultSecret.from_list_secrets_method(list_secret_item)
            if secret.last_version is None:
                continue

            if secret.alias in extra_secrets:
                extra_secrets.pop(extra_secrets.index(secret.alias))
            elif secret.uuid in extra_secrets:
                extra_secrets.pop(extra_secrets.index(secret.uuid))
            elif not args.check_tags(secret.tags):
                continue

            version = VaultVersion.from_get_version_method(secret, client.get_version(secret.last_version))
            versions.append(version)

    if extra_secrets:
        raise ExtraSecretsNotFoundError(', '.join(extra_secrets))

    return versions


def dump_secret_versions(args, versions, compressed=True):
    result = OrderedDict()
    for version in versions:
        result[version.secret.alias] = OrderedDict([
            ('secret', version.secret.uuid),
            ('version', version.uuid),
            ('value', version.value)
        ])

    if compressed:
        return json.dumps(result, separators=(',', ':'))
    else:
        return json.dumps(result, indent=4)


def compress(data):
    compressed = io.BytesIO()
    with gzip.GzipFile(fileobj=compressed, compresslevel=9, mode='wb') as f:
        if six.PY2:
            f.write(data.decode('utf-8'))
        else:
            f.write(bytes(data, 'utf8'))
    return compressed.getvalue()


def has_user_access(version, user):
    for role in version.secret.roles:
        if role.get('login') == user:
            return True
    return False


def handle_print(args):
    versions = gen_versions(args)
    if args.format == 'python':
        for version in versions:
            if args.no_access_from_user:
                if has_user_access(version, args.no_access_from_user):
                    continue
            print(version.secret)
    elif args.format == 'json':
        print(dump_secret_versions(args, versions, compressed=False))


def handle_qloud_export(args):
    versions = gen_versions(args)
    versions_dump = dump_secret_versions(args, versions)
    compressed = compress(versions_dump)

    oneline = base64.standard_b64encode(compressed)
    dmp = []
    while oneline:
        dmp.append(oneline[:args.line_limit])
        oneline = oneline[args.line_limit:]
    base64_dump = '\n'.join(str(d) for d in dmp)

    print(base64_dump)


def handle_file_export(args):
    versions = gen_versions(args)
    secret_dump = dump_secret_versions(args, versions)
    compressed = compress(secret_dump)

    filepath = os.path.abspath(args.file)
    if not os.path.exists(os.path.dirname(filepath)):
        os.makedirs(os.path.dirname(filepath))

    with open(filepath, 'wb') as f:
        f.write(compressed)


def build_tag_expresion(expr):
    if not expr:
        return lambda tags: True

    expr = expr.replace('-', '_')
    try:
        ast_module = ast.parse(expr)
    except SyntaxError as exc:
        raise BadExpressionError(str(exc))

    if len(ast_module.body) != 1 or not isinstance(ast_module.body[0], ast.Expr):
        raise BadExpressionError('Must be only one expression')

    names = []

    def check_node(node):
        if isinstance(node, ast.Expr):
            check_node(node.value)
        elif isinstance(node, ast.BoolOp):
            check_node(node.values[0])
            check_node(node.values[1])
        elif isinstance(node, ast.UnaryOp):
            check_node(node.operand)
        elif isinstance(node, ast.Name):
            names.append(node.id)
        else:
            raise BadExpressionError('Bad node {}'.format(type(node).__name__))

    check_node(ast_module.body[0])

    base_context = {n: False for n in names}

    def check_tags(tags):
        context = base_context.copy()
        for t in tags:
            context[t.replace('-', '_')] = True

        return eval(expr, {}, context)

    return check_tags


def handle_need_secret(args):
    envs = qloud.get_environments_without_secret('rasp', args.secret, token=args.token)
    for e in envs:
        print(e)


def run(args):
    parser = argparse.ArgumentParser(DOC)
    parser.add_argument('-v', '--verbose', action='store_true')

    def add_secret_filter_args(parser):
        parser.add_argument('-t', '--tags', dest='tags', default=None, metavar='TAG_EXPESSION',
                            help='Boolean tag expression, e.g "production or admin"')
        parser.add_argument('-e', '--extra', action='append', default=[],
                            help='Extra secrets by aliases')
        return parser

    subparsers = parser.add_subparsers()

    print_parser = add_secret_filter_args(subparsers.add_parser('print'))
    print_parser.add_argument('--no-access-from-user', help='Без доступа пользователя, для проверки,'
                                                            ' что у пользователя есть прямой доступ.'
                                                            ' Может быть опосредованный доступ через группы.'
                                                            ' Например robot-rasp')
    print_parser.add_argument('--format', choices=('python', 'json'), default='python')
    print_parser.set_defaults(handler=handle_print)

    qloud_parser = add_secret_filter_args(subparsers.add_parser('qloud-export'))
    qloud_parser.set_defaults(handler=handle_qloud_export)
    qloud_parser.add_argument('--line-limit', type=int, default=120)

    file_parser = add_secret_filter_args(subparsers.add_parser('file-export'))
    file_parser.set_defaults(handler=handle_file_export)
    file_parser.add_argument('--file', default='/etc/yav-secrets/rasp.json.gz')

    need_secret_parser = subparsers.add_parser('need-secret',
                                               help='Выводит enviroments, у которых нет указанного секрета')
    need_secret_parser.set_defaults(handler=handle_need_secret)
    need_secret_parser.add_argument('project', metavar='PROJECT')
    need_secret_parser.add_argument('secret', metavar='SECRET')
    need_secret_parser.add_argument('--token', metavar='QLOUD_TOKEN', default=os.getenv('QLOUD_API_TOKEN'))

    args = parser.parse_args(args)

    if args.verbose:
        import logging
        root_logger = logging.getLogger()
        root_logger.setLevel(logging.DEBUG)
        handler = logging.StreamHandler(stream=codecs.getwriter('utf-8')(sys.stdout))
        handler.setFormatter(logging.Formatter('%(levelname)s %(name)s %(message)s'))
        root_logger.addHandler(handler)

    if hasattr(args, 'tags'):
        try:
            args.check_tags = build_tag_expresion(args.tags)
        except CommandError as exc:
            parser.error('{}: {}'.format(type(exc).__name__, str(exc)))

    try:
        args.handler(args)
    except CommandError as exc:
        parser.error('{}: {}'.format(type(exc).__name__, str(exc)))
