# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import namedtuple, OrderedDict

import os
import logging

from library.python.vault_client import instances


logger = logging.getLogger('rasp-vault')


class SecretError(Exception):
    pass


class SecretNotFoundError(SecretError):
    pass


class SecretFetchError(SecretError):
    pass


def get_vault_client(oauth_token=None):
    if oauth_token:
        logger.info('Using passed OAauth token')
        return instances.Production(rsa_auth=False, authorization=oauth_token)
    if os.getenv('RASP_VAULT_OAUTH_TOKEN'):
        logger.info('Using env RASP_VAULT_OAUTH_TOKEN for authentication')
        return instances.Production(rsa_auth=False, authorization=os.getenv('RASP_VAULT_OAUTH_TOKEN'))

    if os.getenv('SSH_AUTH_SOCK'):
        logger.info('Using ssh agent for authentication')
        return instances.Production(rsa_auth=True)

    if os.path.exists(os.path.expanduser('~/.rasp_vault_token')):
        if oct(os.stat(os.path.expanduser('~/.rasp_vault_token')).st_mode)[-3:] != '400':
            raise SecretError('~/.rasp_vault_token must have permissions 400')
        with open(os.path.expanduser('~/.rasp_vault_token')) as f:
            logger.info('Using ~/.rasp_vault_token for authentication')
            return instances.Production(rsa_auth=False, authorization=f.read().strip())

    if os.getenv('YA_TEST_RUNNER') or os.getenv('RASP_VAULT_IGNORE_ERRORS'):
        logger.warn('There is no way to authenticate in vault')
        return None

    raise SecretError('There is no way to authenticate in vault')


class VaultSecret(namedtuple('Secret', ['alias', 'uuid', 'last_version', 'tags', 'roles'])):
    @classmethod
    def from_get_secret_method(cls, get_secret_data):
        return cls(alias=get_secret_data['name'], uuid=get_secret_data['uuid'],
                   last_version=get_secret_data['secret_versions'][0]['version'],
                   tags=tuple(get_secret_data.get('tags', [])),
                   roles=get_secret_data['secret_roles'])

    @classmethod
    def from_list_secrets_method(cls, list_secret_item):
        last_version = list_secret_item.get('last_secret_version', {}).get('version')
        return cls(alias=list_secret_item['name'], uuid=list_secret_item['uuid'],
                   last_version=last_version,
                   tags=tuple(list_secret_item.get('tags', [])),
                   roles=list_secret_item['secret_roles'])

    def __str__(self):
        return '<Secret {self.alias}: {self.uuid} {self.last_version}; {self.tags}>'.format(self=self)


class VaultVersion(namedtuple('Secret', ['secret', 'uuid', 'value'])):
    @classmethod
    def from_get_version_method(cls, secret, get_version_data):
        if secret.uuid != get_version_data['secret_uuid']:
            raise ValueError('Version secret does not match passed secret')

        return cls(secret=secret, uuid=get_version_data['version'],
                   value=OrderedDict(sorted(get_version_data['value'].items())))

    def __str__(self):
        return '<Version of {self.secret.alias}: {self.secret.uuid} {self.uuid}>'.format(self=self)
