# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from collections import OrderedDict

from six import binary_type, iteritems, text_type


class Schematized(object):
    __fields__ = OrderedDict()

    def __init__(self):
        fields = OrderedDict()
        for c in self.get_inheritance_chain()[::-1]:
            fields.update(c.__fields__)
        object.__setattr__(self, '__fields__', fields)
        object.__setattr__(self, '__values__', dict())

    @staticmethod
    def diff(first, second):
        changed = list()
        if type(first) != type(second):
            raise TypeError('Cannot compare {} and {}'.format(type(first), type(second)))
        for key, field in first.__fields__.items():
            first_value = first.__values__.get(key, field.default)
            second_value = second.__values__.get(key, field.default)
            if second_value != first_value:
                changed.append((key, first_value, second_value))
        return changed

    @classmethod
    def get_inheritance_chain(cls):
        inheritance_chain = []
        if cls == Schematized:
            return inheritance_chain
        inheritance_chain.append(cls)
        for c in cls.__bases__[::-1]:
            inheritance_chain.extend(c.get_inheritance_chain())
        return inheritance_chain

    @classmethod
    def from_dict(cls, d, convert_type=False, ignore_unknown=False, encoding='utf8'):
        o = cls()
        for k, v in iteritems(d):
            if not hasattr(o, k) and ignore_unknown:
                continue
            if v is None:
                continue
            if convert_type:
                field = o.__fields__[k]
                if v == '' and field.empty_is_none:
                    continue
                underlying_type = field.get_underlying_type()
                if isinstance(v, binary_type) and underlying_type is text_type:
                    v = v.decode(encoding)
                else:
                    try:
                        v = underlying_type(v)
                    except Exception as e:
                        raise type(e)('Field "{}", failed to convert value "{}": {}'.format(k, v, e))
            o.__setattr__(k, v)
        return o

    def as_dict(self, validate=True):
        if validate:
            self.validate()
        return {k: f.get_value_for_dict(self.__values__.get(k, f.default)) for k, f in iteritems(self.__fields__)}

    def get_hidden_field_names(self):
        return {k for k, v in self.__fields__.items() if v.hidden}

    def get_logfeller_schema(self):
        result = OrderedDict()
        for k, v in self.__fields__.items():
            result[k] = v.get_logfeller_type_name()
        return result

    def get_yt_schema(self):
        result = OrderedDict()
        for k, v in self.__fields__.items():
            result[k] = v.get_yt_type_name()
        return result

    def validate(self):
        not_assigned_fields = list()
        for k, f in iteritems(self.__fields__):
            value = self.__values__.get(k)
            if value is None and not f.optional and f.default is None:
                not_assigned_fields.append(k)
        if not_assigned_fields:
            raise ValueError('Field(s) not assigned: "{}"'.format('", "'.join(not_assigned_fields)))

    def __getattr__(self, item):
        try:
            default = self.__fields__[item].default
            result = self.__values__.get(item, default)
            if result is None:
                result = default
            return result
        except KeyError:
            raise AttributeError('{!r} object has no attribute {!r}'.format(type(self).__name__, item))

    def __setattr__(self, key, value):
        try:
            self.__fields__[key].check(value)
            self.__values__[key] = value
        except KeyError:
            raise AttributeError('{!r} object has no attribute {!r}'.format(type(self).__name__, key))
        except TypeError as e:
            raise TypeError('Failed to set {!r} value, {}'.format(key, e))
