# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from enum import Enum
from six import text_type


class Field(object):
    __logfeller_type_name__ = None
    __yt_type_name__ = None
    __underlying_type__ = None

    def __init__(self, default=None, optional=False, empty_is_none=False, description='', hidden=False):
        self.optional = optional
        self.empty_is_none = empty_is_none
        self.description = description
        if default == '' and self.empty_is_none:
            default = None
        self.check(default)
        self.default = default
        self.hidden = hidden

    def check(self, value):
        self.ensure_type(value, self.__underlying_type__)

    @staticmethod
    def ensure_type(value, type_):
        if value is not None and not isinstance(value, type_):
            raise TypeError('Expected value of {!r} type, but got {!r}'.format(type_.__name__, type(value).__name__))

    @staticmethod
    def get_value_for_dict(value):
        return value

    def get_logfeller_type_name(self):
        return self.__logfeller_type_name__

    def get_yt_type_name(self):
        return self.__yt_type_name__

    def get_underlying_type(self):
        return self.__underlying_type__


class Any(Field):
    __yt_type_name__ = 'any'

    def check(self, value):
        pass

    @staticmethod
    def ensure_type(value, type_):
        pass

    def get_logfeller_type_name(self):
        raise TypeError('You should not send this to logbroker/logfeller')

    def get_underlying_type(cls):
        return lambda x: x


class Boolean(Field):
    __logfeller_type_name__ = 'VT_BOOLEAN'
    __yt_type_name__ = 'boolean'
    __underlying_type__ = bool


class Float(Field):
    __logfeller_type_name__ = 'VT_DOUBLE'
    __yt_type_name__ = 'double'
    __underlying_type__ = float


class Int64(Field):
    __logfeller_type_name__ = 'VT_INT64'
    __yt_type_name__ = 'int64'
    __underlying_type__ = int


class UInt64(Field):
    __logfeller_type_name__ = 'VT_UINT64'
    __yt_type_name__ = 'uint64'
    __underlying_type__ = int

    def check(self, value):
        super(UInt64, self).check(value)
        if value is not None and value < 0:
            raise TypeError('Expected unsigned int value, but got of {!r}'.format(value))


class String(Field):
    __logfeller_type_name__ = 'VT_STRING'
    __yt_type_name__ = 'string'
    __underlying_type__ = text_type


class Enumeration(Field):

    def __init__(self, enum_class, base_field_class, default=None, optional=False, description=''):
        assert issubclass(enum_class, Enum)
        assert issubclass(base_field_class, Field)
        assert not issubclass(base_field_class, Any)

        self.enum_class = enum_class
        self.__logfeller_type_name__ = base_field_class.__logfeller_type_name__
        self.__yt_type_name__ = base_field_class.__yt_type_name__
        self.__underlying_type__ = enum_class

        member = None
        try:
            for member in enum_class:
                self.ensure_type(member.value, base_field_class.__underlying_type__)
        except TypeError as e:
            message = 'Value of {} cannot be represented as {} field. {}'.format(member, base_field_class.__name__, e)
            raise TypeError(message)

        super(Enumeration, self).__init__(default, optional, description=description)

    @staticmethod
    def get_value_for_dict(value):
        if value is None:
            return
        return value.value
