# -*- coding: utf-8 -*-
import json
import logging

import requests
from retrying import retry
from six.moves.urllib.parse import urljoin

from travel.library.python.sender.error import SenderError

log = logging.getLogger(__name__)


class TransactionalApi(object):
    def __init__(self, host, auth, account_slug, campaign_slug):
        self.url = '{host}/api/0/{account_slug}/transactional/{campaign_slug}/'.format(
            host=host,
            account_slug=account_slug,
            campaign_slug=campaign_slug,
        )
        if auth and not len(auth) == 2:
            raise ValueError('Invalid auth type. Should be a tuple of two elements')
        self.auth = auth

    @retry(stop_max_attempt_number=3, wait_fixed=2000)
    def send(self,
             to_email,
             args,
             is_async=True,
             from_name=None,
             from_email=None,
             countdown=None,
             expires=None,
             attachments=None,
             for_testing=False,
             ):
        """

        NOTE: from_email, from_name - (это нужно в редких кейсах; обычно лучше использовать адрес, прописанный в шаблоне Рассылятора)
        :param basestring to_email: куда отправлять. См. также пункт "Отправка на uid"
        :param dict args: json с переменными, которые будут подставлены в тело письма
        :param bool is_async: отправить письмо сразу (синхронно, без очереди) или положить в очередь
        :param from_name: заменить имя отправителя, прописанное в шаблоне
        :param from_email: заменить адрес отправителя, прописанный в шаблоне
        :param basestring|int countdown: для асинхронной отправки - отправить рассылку не сразу, а через countdown секунд.
        :param basestring|int expires: для асинхронной отправки - не отправлять через expires секунд. через какое время уже не нужно отправлять письмо.
        :param attachments: приаттачить эти файлы к письму
        :param for_testing: позволяет отправить пиьсмо, которое не было активировано. Нужно в случае тестирования нового шаблона
        :return:
        """

        if not self.auth or not self.auth[0]:
            log.warning('No auth provided. Message is not sent')
            return None

        data = {
            'to_email': to_email,
            'args': json.dumps(args),
            'async': is_async,
            'from_name': from_name,
            'from_email': from_email,
            'countdown': self._to_sec_str(countdown) if countdown else None,
            'expires': self._to_sec_str(expires) if expires else None,
            'attachments': attachments,
        }

        if for_testing:
            log.warning('Sending email for testing')
            data['for_testing'] = True

        url = urljoin(self.url, 'send')
        r = requests.post(url, auth=self.auth, data=data, verify=False)

        if not r.ok:
            raise SenderError(
                'Send %s failed %r. Sender reponse %s, [%s]' % (
                    url,
                    data,
                    r.status_code,
                    r.content,
                )
            )
        return r

    @staticmethod
    def _to_sec_str(x):
        return '{}s'.format(x)
