# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import threading
import time
from multiprocessing import queues

from flask import Flask, Response, request
from frozendict import frozendict
from werkzeug.serving import make_server

from library.python.monlib.encoder import dumps
from library.python.monlib.metric_registry import MetricRegistry

from travel.library.python.solomon.metrics import GaugeMetric, RateMetric, HistogramRateMetric


CONTENT_TYPE_SPACK = 'application/x-solomon-spack'
CONTENT_TYPE_JSON = 'application/json'


class FlaskThread(threading.Thread):
    def __init__(self, app, host, port):
        super(FlaskThread, self).__init__()
        self.flask_server = make_server(host, port, app)
        self.ctx = app.app_context()
        self.ctx.push()

    def run(self):
        self.flask_server.serve_forever()

    def stop(self):
        self.flask_server.shutdown()


class SolomonMetricsServer(object):
    def __init__(self, host='localhost', port=9947, endpoint='/metrics/', common_labels=None, logger=None):
        self.port = port
        self.host = host
        self.endpoint = endpoint
        self.registry = MetricRegistry(common_labels)

        self.gauges = {}
        self.app = None
        self.web_server_thread = None

        self._metric_queue = None
        self._logger = logger or logging.getLogger(__name__)

    def register_metric_queue(self, metric_queue):
        self._metric_queue = metric_queue

    def register_gauge(self, labels_dict, value_getter):
        labels_dict = frozendict(labels_dict)
        self.gauges[labels_dict] = value_getter

    def process_metric_queue(self):
        if not self._metric_queue:
            return
        now_time = int(time.time())
        while not self._metric_queue.empty():
            metric = self._metric_queue.get()
            if isinstance(metric, RateMetric):
                self.registry.rate(metric.labels).add(metric.value)
            elif isinstance(metric, GaugeMetric):
                self.registry.gauge(metric.labels).set(metric.value)
            elif isinstance(metric, HistogramRateMetric):
                self.registry.histogram_rate(
                    metric.labels,
                    metric.histogram_type,
                    **metric.histogram_options
                ).collect(metric.value)
            else:
                self._logger.error('Unknown metric in metric queue: %s', type(metric))
            if metric.created_at > now_time:
                break

    def update_solomon_data(self):
        self.process_metric_queue()
        for labels_dict, value_getter in self.gauges.items():
            new_value = value_getter()
            self.registry.gauge(dict(labels_dict)).set(new_value)

    def solomon_handler(self):
        self.update_solomon_data()
        if request.headers.get('accept', None) == CONTENT_TYPE_SPACK:
            return Response(dumps(self.registry, format='spack'), mimetype=CONTENT_TYPE_SPACK)
        return Response(dumps(self.registry, format='json'), mimetype=CONTENT_TYPE_JSON)

    def run(self):
        self.app = self._create_app()
        self.app.run(host=self.host, port=self.port)

    def run_in_thread(self):
        self.app = self._create_app()
        self.web_server_thread = FlaskThread(self.app, self.host, self.port)
        self.web_server_thread.start()

    def stop_thread(self):
        if self._metric_queue:
            while True:
                try:
                    self._metric_queue.get_nowait()
                except queues.Empty:
                    break
            self._metric_queue.close()
            self._metric_queue.join_thread()
        if self.web_server_thread and self.web_server_thread.is_alive():
            self.web_server_thread.stop()

    def _create_app(self):
        app = Flask(__name__)
        app.add_url_rule(self.endpoint, view_func=self.solomon_handler)

        return app
