# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from datetime import datetime

from requests import Session
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util import Retry


class SolomonPushClient(object):
    def __init__(self, project, cluster, service, token, ts=None, labels=None):
        if labels is None:
            labels = dict()
        self.labels = labels
        if ts is None:
            ts = datetime.utcnow().isoformat().split(".")[0] + "Z"
        self.ts = ts
        self.params = {
            'project': project,
            'cluster': cluster,
            'service': service
        }
        self.token = token
        self.session = self.get_session()
        self.batch = []

    def send(self, sensor, value, **labels):
        self.batch.append((sensor, value, labels))

    def get_payload(self, batch):
        def get_sensor(name, value, labels):
            sensor_labels = {'sensor': name}
            sensor_labels.update(labels)
            return {
                'labels': sensor_labels,
                'ts': self.ts,
                'value': value,
            }

        common_labels = self.labels.copy()
        sensors = [get_sensor(name, value, labels)
                   for name, value, labels in batch]
        return {
            'commonLabels': common_labels,
            'sensors': sensors,
        }

    def do_request(self, payload):
        url = 'https://solomon.yandex.net/api/v2/push'
        r = self.session.post(url, params=self.params, json=payload)
        r.raise_for_status()

    def get_session(self):
        session = Session()
        retry = Retry(
            total=15,
            method_whitelist=frozenset(['POST']),
            backoff_factor=5.0,
            status_forcelist=frozenset([500, 503, 413, 429])
        )
        session.mount('http://', HTTPAdapter(max_retries=retry))
        session.mount('https://', HTTPAdapter(max_retries=retry))
        if self.token is not None:
            session.headers.update({'Authorization': 'OAuth {}'.format(self.token)})
        return session

    def upload(self):
        if not self.batch:
            return
        payload = self.get_payload(self.batch)
        self.do_request(payload)
        self.batch = []
