# -*- coding: utf-8 -*-

import datetime
from collections import OrderedDict
from typing import Union


class TimeInterval:
    __modifiers__ = OrderedDict([
        ('d', 86400),
        ('h', 3600),
        ('m', 60),
        ('s', 1),
    ])

    def __init__(self, interval: Union[int, str]) -> None:
        if isinstance(interval, int):
            self.interval = interval
        elif isinstance(interval, str):
            self.interval = self.parse(interval)
        else:
            raise Exception(f'This type is not supported for interval: {type(interval)}')

    def __eq__(self, other):
        return self.interval == other.interval

    def __hash__(self):
        return hash(self.interval)

    def __str__(self):
        interval = self.interval
        if interval == 0:
            return '0'
        result = list()
        for modifier, value in self.__modifiers__.items():
            count, interval = divmod(interval, value)
            if count > 0:
                result.append(f'{count}{modifier}')
        return ''.join(result)

    def __repr__(self):
        return str(self)

    def to_timedelta(self) -> datetime.timedelta:
        return datetime.timedelta(seconds=self.interval)

    def parse(self, interval: str) -> int:
        # modifiers may be used in any order but who cares
        interval = interval.strip()
        try:
            return int(interval)
        except ValueError:
            pass
        start = 0
        result = 0
        for i in range(len(interval)):
            s = interval[i]
            value = self.__modifiers__.get(s)
            if value is None:
                continue
            count_str = interval[start:i].strip()
            try:
                count = int(count_str)
            except Exception as e:
                raise Exception(f'Failed to parse "{interval}", {e}')
            result += value * count
            start = i + 1
        return result
