# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from functools import wraps
import logging
import time
import json
import os
import sys


LOG = logging.getLogger(__name__)


def get_size_in_units(size):
    suffixes = ['', 'K', 'M', 'G', 'T', 'P', 'E', 'Z', 'Y']
    for suffix in suffixes:
        if size < 1024:
            break
        size = size / 1024.
    return '{:0.1f} {}'.format(size, suffix)


def with_retries(func=None, max_attempts=5, backoff_factor=1.0, backoff_max=1.0):
    if not func:
        return lambda f: with_retries(f, max_attempts=max_attempts)

    @wraps(func)
    def wrapper(*args, **kwargs):
        attempts = max_attempts
        while True:
            try:
                return func(*args, **kwargs)
            except Exception:
                attempts -= 1
                if attempts == 0:
                    raise
                LOG.exception('%r failed, %d attempts left' % (func, attempts))
                sleep_time = backoff_factor * 2 ** (max_attempts - attempts - 1)
                sleep_time = min(sleep_time, backoff_max)
                time.sleep(sleep_time)
    return wrapper


DEFAULT_ARG_REPLACES_ENV = 'ARG_REPLACES'


def replace_args_from_env(args=None, env_key=DEFAULT_ARG_REPLACES_ENV):
    if args is None:
        args = sys.argv[1:]
    env_args = os.environ.get(env_key)
    if env_args is None:
        return args
    replaces = json.loads(env_args)
    return replace_args(args, replaces)


def replace_args(args, replaces):
    result_args = []
    for arg in args:
        for k, v in replaces.items():
            arg = arg.replace(k, v)
        result_args.append(arg)
    return result_args
