# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from functools import wraps

from django_opentracing import DjangoTracing
from opentracing import global_tracer

from travel.library.python.tracing import global_context


def traced_view(*attributes):
    """
    Decorator for Django views
    :param attributes:
    :return:
    """
    def decorator(view_func):
        def wrapper(request, *args, **kwargs):
            return view_func(request)

        wrapper.__name__ = view_func.__name__
        return wrapper

    tracing = Tracing.instance
    if hasattr(tracing, 'trace'):
        return tracing.trace(*attributes)
    return decorator


class Tracing(DjangoTracing):
    instance = None

    def __getattr__(self, name):
        return getattr(self.tracer, name)

    def __new__(cls, tracer):
        if cls.instance is None:
            cls.instance = super(Tracing, cls).__new__(cls)
        return cls.instance

    def trace(self, *attributes):
        """
        Function decorator that fixes decorator from django-opentracing
        that traces functions such as Views
        @param attributes any number of HttpRequest attributes
        (strings) to be set as tags on the created span
        """
        def decorator(view_func):

            @wraps(view_func)
            def wrapper(request, *args, **kwargs):
                # if tracing all already, return right away.
                if self._trace_all:
                    return view_func(request)

                # otherwise, apply tracing.
                try:
                    self._apply_tracing(request, view_func, list(attributes))
                    r = view_func(request, *args, **kwargs)
                except Exception as exc:
                    self._finish_tracing(request, error=exc)
                    raise

                self._finish_tracing(request, r)
                return r

            return wrapper

        return decorator


class TracingMiddleware(object):
    """
    Django tracing middleware.
    Takes reqid and puts it into tags, also adds global_context to span.
    """
    def process_view(self, request, view_func, view_args, view_kwargs):
        tracer = global_tracer()
        active_span = getattr(tracer.scope_manager.active, 'span', None)
        if active_span is not None:
            active_span.set_tag('reqid', request.GET.get('reqid'))
            global_context.add_context_to_span(active_span)

            with tracer.scope_manager.activate(active_span, True):
                return view_func(request, *view_args, **view_kwargs)
