# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import threading
from functools import partial

import flask
import opentracing
import tornado
import six
from flask_opentracing import FlaskTracing
from jaeger_client import Config
from jaeger_client.constants import SAMPLER_TYPE_PROBABILISTIC
from opentracing.scope_managers.tornado import TornadoScopeManager
from opentracing_instrumentation.client_hooks import boto3, install_all_patches


class FlaskTracingWithFilter(FlaskTracing):
    """
    Класс нужен для фильтрации того, что попадает в трассировку.
    Например, обычно нужно исключить 'ping'.
    """
    def __init__(self, tracer=None, trace_all_requests=None, app=None,
                 traced_attributes=(), start_span_cb=None, filter_paths=()):
        self.filter_paths = filter_paths
        super(FlaskTracingWithFilter, self).__init__(tracer, trace_all_requests, app,
                                                     list(traced_attributes), start_span_cb)

    def _before_request_fn(self, traced_attributes):
        if flask._request_ctx_stack.top.request.endpoint in self.filter_paths:
            return

        super(FlaskTracingWithFilter, self)._before_request_fn(traced_attributes)


def install_all_working_patches(scope_manager):
    """
    https://st.yandex-team.ru/RASPTICKETS-17918
    Patch of boto3 is compatible only with TornadoScopeManager
    """
    install_all_patches()
    if not isinstance(scope_manager, TornadoScopeManager):
        boto3.reset_patches()


def setup_tracing(
    flask_application,
    service_name,
    version='unknown',
    include_env_variables=(),
    start_span_call_back=None,
    traced_attributes_of_flask_request=None,
    run_tornado_in_thread=True,
    filter_paths=(),
    instrument_libraries_with_tracing=True,
    scope_manager=None,
    jaeger_sampler_type=SAMPLER_TYPE_PROBABILISTIC,
    jaeger_sampler_parameter=0.001,
    jaeger_client_enable_logging=True,
):
    """
    :param Flask flask_application: объект flask-приложения
    :param str service_name: название трассируемого проекта, отображается на фронтенде егеря
    :param str version: версия трассируемого проекта
    :param list[str] include_env_variables: переменные окружения, которые нужно сделать тегами
    :param callable start_span_call_back: колбэк для возможности добавить свои теги,
                                          принимает два аргумента span и request
    :param list[str] traced_attributes_of_flask_request: атрибуты flask.Request, которые попадут в теги
    :param bool run_tornado_in_thread: запустить торнадо (обычно нужно для отправки трассировок в jaeger_agent)
    :param list[str] filter_paths: пути, которые не надо трассировать, например, ['ping']
    :param bool instrument_libraries_with_tracing: прокидывать трассировку в вызовы requests и других библиотек
    :param ScopeManager scope_manager: объект класса ScopeManager, при None будет ThreadLocalScopeManager
    :param str jaeger_sampler_type: параметр для настройки трассировки,
                                    возможные значения const, probabilistic, ratelimiting, remote
    :param float jaeger_sampler_parameter: параметр для настройки трассировки
    :param bool jaeger_client_enable_logging: иногда может понадобится отключить логирование
                                              https://st.yandex-team.ru/TRAINS-5051#5e93197ed799081de77d1ea2

    Параметры для настройки трассировки https://www.jaegertracing.io/docs/1.14/sampling/#client-sampling-configuration
    """
    if traced_attributes_of_flask_request is None:
        traced_attributes_of_flask_request = ('path', 'args', 'data', 'form', 'host', 'remote_addr')

    if instrument_libraries_with_tracing:
        install_all_working_patches(scope_manager)

    jaeger_config = Config(
        config={
            'sampler': {
                'type': jaeger_sampler_type,
                'param': jaeger_sampler_parameter,
            },
            'logging': jaeger_client_enable_logging,
        },
        service_name=service_name,
        scope_manager=scope_manager,
        validate=True,
    )

    # Если это повторная инициализация трассировки, то вернется None
    opentracing_tracer = jaeger_config.initialize_tracer()
    if opentracing_tracer is None:
        opentracing_tracer = opentracing.tracer
    else:
        opentracing.set_global_tracer(opentracing_tracer)

    FlaskTracingWithFilter(
        opentracing_tracer,
        trace_all_requests=True,
        app=flask_application,
        traced_attributes=(),
        start_span_cb=partial(
            _start_span_call_back,
            version=version,
            include_env_variables=include_env_variables,
            traced_attributes_of_flask_request=traced_attributes_of_flask_request,
            start_span_call_back=start_span_call_back
        ),
        filter_paths=filter_paths
    )

    if run_tornado_in_thread:
        thread = threading.Thread(target=lambda: tornado.ioloop.IOLoop.current().start())
        thread.setDaemon(True)
        thread.start()


def _start_span_call_back(span, request, version, include_env_variables,
                          traced_attributes_of_flask_request, start_span_call_back):
    span.set_tag('version', version)

    for env_name in include_env_variables:
        value = os.getenv(env_name)
        if value:
            span.set_tag(env_name, value)

    # fix error in Flask-Opentracing https://st.yandex-team.ru/RASPTICKETS-17962
    for attr in traced_attributes_of_flask_request:
        if hasattr(request, attr):
            payload = getattr(request, attr)
            if payload not in ('', b''):  # python3
                span.set_tag(attr, six.text_type(payload))

    if start_span_call_back is not None:
        try:
            start_span_call_back(span, request)
        except Exception:
            pass
