# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import threading
import tornado

import opentracing
from opentracing_instrumentation.client_hooks.requests import patcher

from travel.library.python.tracing import global_context


def _add_global_context_hook(response, span):
    global_context.add_context_to_span(span)


def install_all_patches():
    """
    Copy of opentracing_instrumentation.client_hooks.install_all_patches
    but only the patches which work with our initialization
    """
    # from opentracing_instrumentation.client_hooks import boto3  this path doesn't work now
    from opentracing_instrumentation.client_hooks import celery
    from opentracing_instrumentation.client_hooks import mysqldb
    from opentracing_instrumentation.client_hooks import psycopg2
    from opentracing_instrumentation.client_hooks import strict_redis
    from opentracing_instrumentation.client_hooks import sqlalchemy
    from opentracing_instrumentation.client_hooks import tornado_http
    from opentracing_instrumentation.client_hooks import urllib
    from opentracing_instrumentation.client_hooks import urllib2
    from opentracing_instrumentation.client_hooks import requests

    celery.install_patches()
    mysqldb.install_patches()
    psycopg2.install_patches()
    strict_redis.install_patches()
    sqlalchemy.install_patches()
    tornado_http.install_patches()
    urllib.install_patches()
    urllib2.install_patches()
    requests.install_patches()


def build_post_fork(jaeger_config, wrapper_cls=None, func=None, patch_all=True):
    """
    Builds gunicorn post_fork hook. Hook is used to initialize tracer, apply opentracing_instrumentation patches
    (if needed) and start tornado IOLoop. Last is necessary as it will be stopped in fork, thus making impossible to
    send spans to jaeger.

    :param jaeger_config: JaegerConfig to be used
    :param wrapper_cls: custom tracer wrapper, for ex. DjangoTracer
    :param func: some callback to be called after initialization
    :param patch_all: wether apply all opentracing_instrumentation patches or not, by default True
    :return:
    """
    is_tracing_on = os.environ.get('TRACING_IS_TRACING_ON', False)
    if not is_tracing_on:
        return lambda server, worker: None

    def post_fork(server, worker):
        # Если это повторная инициализация трассировки, то вернется None
        opentracing_tracer = jaeger_config.initialize_tracer()
        if opentracing_tracer is None:
            opentracing_tracer = opentracing.tracer
        else:
            opentracing.set_global_tracer(opentracing_tracer)

        if wrapper_cls is not None:
            wrapper_cls(opentracing_tracer)

        global_context.set_qloud_image_version()
        patcher.set_response_handler_hook(_add_global_context_hook)

        if patch_all:
            install_all_patches()

        thread = threading.Thread(target=lambda: tornado.ioloop.IOLoop.current().start(), name="jaeger_tornado_loop:{}".format(os.getpid()))
        thread.setDaemon(True)
        thread.start()

        if func is not None:
            func()

    return post_fork
