# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from contextlib import contextmanager
from functools import partial, wraps

import opentracing
from opentracing_instrumentation import get_current_span
from opentracing_instrumentation.utils import start_child_span


@contextmanager
def child_span(name, tracer=None):
    if tracer is None:
        tracer = opentracing.global_tracer()

    if tracer is not None:
        span = tracer.start_active_span(name)
        try:
            yield span
        finally:
            if span is not None:
                span.close()


def traced_function(func=None, name=None, require_active_trace=False):
    """
    A decorator that enables tracing of the wrapped function
    It's a lite version of traced_function from opentracing_instrumentation
    """
    if func is None:
        return partial(
            traced_function,
            name=name,
            require_active_trace=require_active_trace
        )

    if name:
        operation_name = name
    else:
        # TODO: remove prefix from func.__module__, make it beautiful
        operation_name = '%s.%s' % (func.__module__, func.__name__)

    @wraps(func)
    def wrapper(*args, **kwargs):
        parent_span = get_current_span()
        if parent_span is None and require_active_trace:
            return func(*args, **kwargs)

        span = start_child_span(
            operation_name=operation_name, parent=parent_span)
        try:
            res = func(*args, **kwargs)
            span.finish()
            return res
        except Exception as e:
            span.log(event='exception', payload=e)
            span.set_tag('error', 'true')
            span.finish()
            raise
        return func(*args, **kwargs)

    return wrapper
