# coding: utf-8
import os
from functools import partial

import opentracing
import tornado_opentracing
from jaeger_client import Config
from jaeger_client.constants import SAMPLER_TYPE_PROBABILISTIC
from opentracing.scope_managers.tornado import TornadoScopeManager
from opentracing_instrumentation.client_hooks import install_all_patches


class TornadoTracingWithFilter(tornado_opentracing.TornadoTracing):
    """
    Класс нужен для фильтрации того, что попадает в трассировку.
    Например, обычно нужно исключить 'ping'.
    """
    def __init__(self, tracer=None, start_span_cb=None, filter_paths=()):
        self.filter_paths = filter_paths
        super(TornadoTracingWithFilter, self).__init__(tracer, start_span_cb)

    def _apply_tracing(self, handler, attributes):
        if handler.request.path in self.filter_paths:
            return

        return super(TornadoTracingWithFilter, self)._apply_tracing(handler, attributes)


def setup_tracing(service_name, version='unknown', include_env_variables=(), start_span_call_back=None,
                  traced_attributes_of_tornado_request=None, filter_paths=(),
                  jaeger_sampler_type=SAMPLER_TYPE_PROBABILISTIC, jaeger_sampler_parameter=0.001):
    """
    :param str service_name: название трассируемого проекта, отображается на фронтенде егеря
    :param str version: версия трассируемого проекта
    :param callable start_span_call_back: колбэк для возможности добавить свои теги,
                                          принимает два аргумента span и request
    :param list[str] include_env_variables: переменные окружения, которые нужно сделать тегами
    :param list[str] traced_attributes_of_tornado_request: атрибуты tornado.httputil.HTTPServerRequest,
                                                           которые попадут в теги
    :param list[str] filter_paths: пути, которые не надо трассировать, например, ['/ping']
    :param str jaeger_sampler_type: параметр для настройки трассировки,
                                    возможные значения const, probabilistic, ratelimiting, remote
    :param float jaeger_sampler_parameter: параметр для настройки трассировки

    Параметры для настройки трассировки https://www.jaegertracing.io/docs/1.14/sampling/#client-sampling-configuration
    """
    if traced_attributes_of_tornado_request is None:
        traced_attributes_of_tornado_request = ('method', 'path', 'query', 'query_arguments', 'body_arguments')
    install_all_patches()  # прокидывает трассировку в вызовы requests и других библиотек
    jaeger_config = Config(
        config={
            'sampler': {
                'type': jaeger_sampler_type,
                'param': jaeger_sampler_parameter,
            },
            'logging': True,
        },
        service_name=service_name,
        validate=True,
    )

    # Если это повторная инициализация трассировки, то вернется None
    opentracing_tracer = jaeger_config.initialize_tracer()
    if opentracing_tracer is None:
        opentracing_tracer = opentracing.tracer
    else:
        opentracing_tracer._scope_manager = TornadoScopeManager()
        opentracing.set_global_tracer(opentracing_tracer)

    tornado_opentracing.init_tracing()

    tornado_application_tracing_kwargs = {
        'opentracing_tracing': TornadoTracingWithFilter(opentracing_tracer, filter_paths=filter_paths),
        'opentracing_trace_client': False,  # install_all_patches лучше реализует прокидывание спанов
        'opentracing_start_span_cb': partial(
            _start_span_call_back,
            version=version,
            include_env_variables=include_env_variables,
            start_span_call_back=start_span_call_back
        ),
        'opentracing_traced_attributes': traced_attributes_of_tornado_request
    }
    return tornado_application_tracing_kwargs


def _start_span_call_back(span, request, version, include_env_variables, start_span_call_back):
    span.set_tag('version', version)

    for env_name in include_env_variables:
        value = os.getenv(env_name)
        if value:
            span.set_tag(env_name, value)

    if start_span_call_back is not None:
        try:
            start_span_call_back(span, request)
        except Exception:
            pass
