from contextlib import contextmanager
from functools import partial, wraps

from django_opentracing import DjangoTracing
from opentracing.mocktracer import MockTracer
from opentracing_instrumentation import get_current_span
from opentracing_instrumentation.utils import start_child_span


class DummySpan:
    @staticmethod
    def set_tag(name, value):
        pass


class DummyTracing:
    _trace_all = False

    @staticmethod
    def trace():
        def decorator(view_func):
            def wrapper(request, *args, **kwargs):
                return view_func(request)

            wrapper.__name__ = view_func.__name__
            return wrapper
        return decorator

    @staticmethod
    def get_span(request):
        return DummySpan

    tracer = MockTracer()


class _GlobalTracing:
    tracing = DummyTracing
    image_version = None

    def __new__(cls):
        if not hasattr(cls, 'instance'):
            cls.instance = super(_GlobalTracing, cls).__new__(cls)
        return cls.instance

    def __getattr__(self, name):
        return getattr(self.tracing, name)


_tracing = _GlobalTracing()


@contextmanager
def child_span(name, current_tracing=None):
    if current_tracing is None:
        current_tracing = _tracing.tracing

    if current_tracing is not None:
        span = current_tracing.tracer.start_active_span(name)
        try:
            yield span
        finally:
            if span is not None:
                span.close()


def traced_view(*attributes):
    return _tracing.trace(*attributes)


def traced_function(func=None, name=None, require_active_trace=False):
    """
    A decorator that enables tracing of the wrapped function
    It's a lite verxion of traced_function from opentracing_instrumentation
    """
    if func is None:
        return partial(
            traced_function,
            name=name,
            require_active_trace=require_active_trace
        )

    if name:
        operation_name = name
    else:
        # TODO: remove prefix from func.__module__, make it beautiful
        operation_name = '%s.%s' % (func.__module__, func.__name__)

    @wraps(func)
    def wrapper(*args, **kwargs):
        parent_span = get_current_span()
        if parent_span is None and require_active_trace:
            return func(*args, **kwargs)

        span = start_child_span(
            operation_name=operation_name, parent=parent_span)
        try:
            res = func(*args, **kwargs)
            span.finish()
            return res
        except Exception as e:
            span.log(event='exception', payload=e)
            span.set_tag('error', 'true')
            span.finish()
            raise
        return func(*args, **kwargs)

    return wrapper


class Tracing(DjangoTracing):
    def trace(self, *attributes):
        """
        Function decorator that fix decorator from django-opentracing
        that traces functions such as Views
        @param attributes any number of HttpRequest attributes
        (strings) to be set as tags on the created span
        """
        def decorator(view_func):

            @wraps(view_func)
            def wrapper(request, *args, **kwargs):
                # if tracing all already, return right away.
                if self._trace_all:
                    return view_func(request)

                # otherwise, apply tracing.
                try:
                    self._apply_tracing(request, view_func, list(attributes))
                    r = view_func(request, *args, **kwargs)
                except Exception as exc:
                    self._finish_tracing(request, error=exc)
                    raise

                self._finish_tracing(request, r)
                return r

            return wrapper

        return decorator
